<?php

    namespace DPP\Security\Cryptography\Algorithm\HMAC;

    use DPP\Security\Cryptography\Algorithm\Exceptions\InvalidKeyLengthException;

    /**
     * A trait for enabling support of the "HS384" algorithm.
     */
    trait HS384
    {
        /**
         * Encode given data using a given key.
         *
         * @param  mixed  $key
         * @param  string $dataToSign
         * @return string|null
         */
        protected function encodeWithHS384($key, $dataToSign)
        {
            $this->checkKeyLength ($key);

            return hash_hmac('SHA384', $dataToSign, $key, true);
        }

        /**
         * Verify a given signature using a given key.
         *
         * @param  mixed  $key
         * @param  string $dataToSign
         * @param  string $signature
         * @return bool
         */
        protected function verifyWithHS384($key, $dataToSign, $signature)
        {
            $this->checkKeyLength ($key);

            return hash_equals($signature, $this->encodeWithHS384($key, $dataToSign));
        }

        /**
         * Verify a given key length.
         *
         * @param  mixed  $key
         * @param  string $dataToSign
         * @param  string $signature
         * @return bool
         * @throws InvalidKeyLengthException
         */
        private static function checkKeyLength($key) {
            if (strlen(bin2hex($key)) < 25) {
                throw new InvalidKeyLengthException('Must be 24 characters or longer.');
            }
        }
    }