<?php

    namespace DPP\Security\Cryptography\Algorithm\HMAC;

    use DPP\Security\Cryptography\Algorithm\Exceptions\InvalidKeyLengthException;

    /**
     * A trait for enabling support of the "HS256" algorithm.
     */
    trait HS256
    {
        /**
         * Encode given data using a given key.
         *
         * @param  mixed  $key
         * @param  string $dataToSign
         * @return string|null
         */
        protected static function encodeWithHS256($key, $dataToSign)
        {
            self::checkKeyLength ($key);

            return hash_hmac('SHA256'
                            , $dataToSign
                            , $key
                            , true
                            );
        }

        /**
         * Verify a given signature using a given key.
         *
         * @param  mixed  $key
         * @param  string $dataToSign
         * @param  string $signature
         * @return bool
         */
        protected static function verifyWithHS256($key, $dataToSign, $signature)
        {
            self::checkKeyLength ($key);

            return hash_equals($signature, self::encodeWithHS256($key, $dataToSign));
        }

        /**
         * Verify a given key length.
         *
         * @param  mixed  $key
         * @param  string $dataToSign
         * @param  string $signature
         * @return bool
         * @throws InvalidKeyLengthException
         */
        private static function checkKeyLength($key) {
            if (strlen(bin2hex($key)) < 25) {
                throw new InvalidKeyLengthException('Must be 24 characters or longer.');
            }
        }
    }