An implementation of the [JSON Web Token (JWT)](https://tools.ietf.org/html/draft-ietf-oauth-json-web-token-30) draft in PHP. See [jwt.io](http://jwt.io/) for more information on JWT.

Features include:

- Token serialization
- Token deserialization
- Token verification
  - `aud`, `exp`, `iss`, `nbf`, `sub` claims
- Symmetric Encryption Algorithms
  - `HS256`
- Asymmetric Encryption Algorithms
  - none as of yet

- Not support - and won't be
  - Symmetric Encryption Algorithms
    - `NONE` algorithms

- Future inclusion
  - Token verification
    - `kid` claims
  - Symmetric Encryption Algorithms
    - `HS384`, `HS512`
  - Asymmetric Encryption
    - `RS256`, `RS384`, `RS512`
    - `ES256`, `ES384`, `ES512`
    - `PS256`, `PS384`, `PS512`

This library is not susceptible to a common [encryption vulnerability](https://auth0.com/blog/2015/03/31/critical-vulnerabilities-in-json-web-token-libraries/).
[as best I can tell!]

## Installation

```
composer require dpp/security
```

## Usage

Create an instance of the `Emarref\Jwt\Token` class, then configure it.

Define a payload. This is "private data" that is not part of the RFC definition.
As an example...
```php
    $payload = (object) [
          'id'   => '' // whatever you desire
        , 'role' => '' // whatever you desire
    ];
```


```php
use DPP\Security\Cryptography\JWT\JWT;

    /** @var JWT $jwtObject */
    $jwtObject = new JWT();
    $jwtObject->setDataClaim($payloadData)  // <-- defined before hand
              ->setExpiry(JWT_EXPIRY)       // CONST defined in 'credentials file'
              ->setAudClaim('crm')          // to whom this token is for
              ->setAlgorithm('HS256')       // which algorithm is to be used. HS256 is the only one available
              ->setSecretKey(JWT_SECRET);   // CONST defined in 'credentials file'
```

To retrieve the encoded token for transfer...

```php
   $token = (string)$jwtObject;
```

The `$token` variable now contains the encrypted base64 encoded string representation of your token.

To verify a token's...
```php
    // $jwt is the string pulled form HEADER
    if(strpos($_SERVER['HTTP_AUTHORIZATION'], 'Bearer') === 0) {
        list($x, $jwtToken) = explode(' ', $_SERVER['HTTP_AUTHORIZATION']);
    }

    try {
       Validate::token($jwtToken, JWT_SECRET);
    } catch(VerificationException $e) {
       echo $e->getMessage();
    }
```

To retrieve the "data" Claims segment of anm inbound token...
```php
    /** @var array $jwt_payload */
    $jwt_payload = Validate::getDataClaim($jwtToken);
```


## Testing

This library will PHPUnit for unit testing...
 
Eventual.

## Further Reading

- [JSON Web Encryption (JWE)](https://tools.ietf.org/html/draft-ietf-jose-json-web-encryption-36)
- [JSON Web Signature (JWS)](https://tools.ietf.org/html/draft-ietf-jose-json-web-signature-36)
- [JSON Web Algorithms (JWA)](https://tools.ietf.org/html/draft-ietf-jose-json-web-algorithms-36)
- [JSON Web Key (JWK)](https://tools.ietf.org/html/draft-ietf-jose-json-web-key-36)
- [http://self-issued.info/docs/draft-ietf-oauth-json-web-token.html](http://self-issued.info/docs/draft-ietf-oauth-json-web-token.html)
