<?php

    namespace DPP\Security\Cryptography\JWT;

    use DateTimeImmutable;

    use DPP\Security\Cryptography\Algorithm\HMAC\HS256;
    use DPP\Security\Cryptography\Algorithm\HMAC\HS384;
    use DPP\Security\Cryptography\Algorithm\HMAC\HS512;

    use DPP\Security\Cryptography\JWT\Interfaces\Claims as ClaimsInterface;
    use DPP\Security\Cryptography\JWT\Exceptions\JwtException;
    use DPP\Security\Encoders\B64Encoder;
    use DPP\Security\Encoders\JsonEncoder;
    use Exception;

    /**
     * JSON Web Token implementation
     * @link https://tools.ietf.org/html/rfc7519
     *
     * Ideas and code borrowed form multiple sources
     *
     * PHP version 5+
     *
     * @category Authentication
     * @package  Authentication_JWT
     * @author   Walter Torres <w.torres@debtpaygateway.com>
     */
    class JWT implements ClaimsInterface
    {

        use tGenerate;

        use HS256;
//        use HS384;
//        use HS512;
//
//        use RS256;
//        use RS384;
//        use RS512;

        const ALGORITHM_HS256 = 'HS256';
        const ALGORITHM_HS384 = 'HS384';
        const ALGORITHM_HS512 = 'HS512';
        const ALGORITHM_RS256 = 'RS256';
        const ALGORITHM_RS384 = 'RS384';
        const ALGORITHM_RS512 = 'RS512';

        /**
         * Default "type" of JWT to be created, usually just JWT.
         *
         * @link https://tools.ietf.org/html/rfc7519#section-5.1
         *
         * @var string
         */
        const DEFAULT_TYPE = 'JWT';

        /**
         * Default algorithm to use when encoding token when algorithm is provided
         * in token header nor as parameter to encode method.
         */
        const DEFAULT_ALGORITHM = self::ALGORITHM_HS256;

        /**
         * Leeway time (in seconds) to account for clock skew between servers.
         *
         * @var int
         */
        const DEFAULT_LEEWAY = 5;


// *******************************************************************
// Class methods

        /**
         * JWT constructor.
         *
         * @param bool $algorithm
         */
        public function __construct() {
            // Nuke from Orbit, just to be sure
            $this->reset();

            // Json Token Id: an unique identifier for this particular token
            $this->setJwtID();

            // Issued At Time: time when the token was generated
            // This can be altered manually, but it really should be set
            // automatically
            $this->setIatClaim();
        }

        /**
         * Get the string representation of the JWT (i.e. its hash).
         *
         * @return string
         * @throws JwtException
         */
        public function __toString()
        {
            try {
                return (string) $this->generate();
            }

            catch (Exception $e) {
                return 'Could not generate Token: ' . $e->getMessage();
            }

        }

        // General methods **********************************************

        /**
         * Reset the internal state of the JWT object.
         *
         * @return JWT
         */
        public function reset()
        {
            $this->clearHeaders();
            $this->clearPayloadClaims();

            // Leeway to default value
            $this->setLeeway();

            // ISSuer of token
            $this->setIssClaim();

             return $this;
        }


    }
