<?php

    namespace DPP\Security\Cryptography\JWT;


    use DPP\Security\Cryptography\JWT\Exceptions\BeforeValidException;
    use DPP\Security\Cryptography\JWT\Exceptions\InvalidJwtException;
    use DPP\Security\Cryptography\JWT\Exceptions\TokenExpiredException;
    use DPP\Security\Cryptography\JWT\Exceptions\UndefinedTokenException;
    use DPP\Security\Encoders\B64Encoder;

    use DPP\Security\Encoders\Exceptions\JsonDecodingException;
    use DPP\Security\Encoders\JsonEncoder;

    trait tClaims
    {

        /**
         * Holds the JWT payload claims.
         *
         * @link https://tools.ietf.org/html/rfc7519#section-4.1
         *
         * @var object
         */
        private static $claims;
        private static $claimsJson;

        /**
         * Defines the expiry date/time
         * 15 minutes from current time
         *
         * @link https://tools.ietf.org/html/rfc7519#section-4.1.3
         *
         * @var string
         */
        private static $expiry;

        /** Default value, in minutes */
        private static $default_expiry = 15;

// *******************************************************************
// Class methods

        /**
         * Set a custom payload claim on the JWT.
         * The RFC calls these private claims.
         * 
         * Eg: you may wish to set a user_id or a username in the JWT payload
         *
         * @param string $claimName
         * @param mixed  $claimValue
         *
         * @return void
         */
        protected static function setPayloadClaim ($claimName, $claimValue)
        {
            $claimName = strtolower($claimName);
            self::$claims->$claimName = $claimValue;
        }

        /**
         * Retrieves payload claim on the JWT.
         *
         * @param string $claimName
         *
         * @return string|boolean
         */
        protected static function getPayloadClaim ($claimName)
        {
            if (array_key_exists(JWT::ISSUED_AT, self::$claims)) {
                $claimValue = self::$claims->$claimName;
            } else {
                $claimValue = false;
            }

            return $claimValue;
        }

        /**
         * remove a claim from current definition.
         *
         * @param string $claimName
         *
         * @return void
         */
        protected function unsetPayloadClaim ($claimName)
        {
            unset(self::$claims[$claimName]);
        }

        /**
         * remove entire claims definition.
         *
         * @return void
         */
        protected function clearPayloadClaims ()
        {
            self::$claims = (object)[
                  JWT::ISSUER      => false  // ISSuer
                , JWT::SUBJECT     => false  // Subject
                , JWT::AUDIENCE    => false  // Audience
                , JWT::EXPIRY      => false  // EXPires
                , JWT::NOT_BEFORE  => false  // Not BeFore
                , JWT::ISSUED_AT   => false  // Issued At Time: time when the token was generated
                , JWT::JID         => false  // Json Token Id: an unique identifier for the token
                , 'data' => false  // DATA related to the signer user
            ];
        }

        // ISS: Issuer
        /**
         * Set the Issuer [iss] JWT payload claim.
         * This defines who issued the token.
         *
         * Can be a string or URI.
         *
         * @link https://tools.ietf.org/html/rfc7519#section-4.1.1
         *
         * @param string|bool $iss
         *
         * @return JWT
         */
        final public function setIssClaim($iss = false)
        {
            if($iss === false) {
                /** @var string $iss */
                $iss = (@$_SERVER['SERVER_NAME'] ?: 'cli');
            }

            $this->setPayloadClaim(JWT::ISSUER, $iss);

            return $this;
        }

        /**
         * Retrieve the Issuer [iss] JWT payload claim.
         *
         * Can be a string or URI.
         *
         * @link https://tools.ietf.org/html/rfc7519#section-4.1.1
         *
         * @return bool|string
         */
        protected function getIssClaim()
        {
            return $this->getPayloadClaim(JWT::ISSUER);
        }

        /**
         * Validate the "iss" (issuer) claim.
         *
         * @return bool
         * @throws InvalidClaimTypeException
         */
        protected function validateIssClaim()
        {
            $iss = $this->getPayloadClaim (JWT::ISSUER);

            if ($iss) {
                Validation::checkClaimType(JWT::ISSUER, $iss, 'string');
            }

            return true;
        }

        // SUB: Subject
        /**
         * Set the Subject [sub] JWT payload claim.
         * This defines the principal subject for the token.
         *
         * Can be a string or URI.
         *
         * @link https://tools.ietf.org/html/rfc7519#section-4.1.2
         *
         * @param string|bool $sub
         *
         * @return JWT
         */
        final public function setSubClaim($sub = false)
        {
            $this->setPayloadClaim(JWT::SUBJECT, $sub);

            return $this;
        }

        /**
         * Retrieve the Subject [sub] JWT payload claim.
         *
         * Can be a string or URI.
         *
         * @link https://tools.ietf.org/html/rfc7519#section-4.1.2
         *
         * @param string|bool $sub
         *
         * @return bool|string
         */
        protected function getSubClaim()
        {
            return $this->getPayloadClaim(JWT::SUBJECT);
        }

        // AUD: Audience
        /**
         * Set the Audience [aud] JWT payload claim.
         * This defines to whom the token is issued for.
         *
         * Can be a string or URI.
         *
         * @link https://tools.ietf.org/html/rfc7519#section-4.1.3
         *
         * @param string|bool $aud
         *
         * @return JWT
         */
        final public function setAudClaim($aud = false)
        {
            $this->setPayloadClaim(JWT::AUDIENCE, $aud);

            return $this;
        }

        /**
         * Retrieves the Audience [aud] JWT payload claim.
         *
         * @link https://tools.ietf.org/html/rfc7519#section-4.1.3
         *
         * @return bool|string
         */
        protected function getAudClaim()
        {
            return $this->getPayloadClaim(JWT::AUDIENCE);
        }

        /**
         * Validate the "aud" (audience) claim, if included.
         *
         * @param  string|string[] $audience
         * @return void
         * @throws InvalidJwtException
         */
        protected function validateAudClaim($audience)
        {
            if (array_key_exists('aud', self::$claims)) {
                $audience = is_array($audience) ? $audience : [$audience];

                // Make sure the audience claim is set to a valid value
                $foundAudience = is_array(self::$claims[JWT::AUDIENCE]) ? self::$claims[JWT::AUDIENCE] : [self::$claims[JWT::AUDIENCE]];
                $expectedIndex = 0;

                foreach ($foundAudience as $index => $value) {
                    if (is_string($value) && $index === $expectedIndex) {
                        $expectedIndex++;
                    } else {
                        throw new InvalidJwtException('Invalid "aud" value.', $this);
                    }
                }

                // Make sure the JWT is intended for any of the expected audiences
                foreach ($audience as $expectedAudience) {
                    if (in_array($expectedAudience, $foundAudience, true)) {
                        return;
                    }
                }

                throw new InvalidJwtException('Invalid JWT audience.', $this);
            }
        }

        // EXP: Expiration Time
        /**
         * Set the Expiration [exp] JWT payload claim.
         * This sets the time at which the JWT token will expire and
         * is no longer be accepted.
         *
         * This expects UNIX TIMESTAMP[int] value
         *
         * @link https://tools.ietf.org/html/rfc7519#section-4.1.4
         *
         * @param int $expiry   UNIX TIMESTAMP[int] value
         *
         * @return JWT
         * @throws \Exception
         */
        public function setExpClaim($expiry)
        {
            self::setPayloadClaim(JWT::EXPIRY, $expiry);

            return $this;
        }

        /**
         * Retrieves the Expiration [exp] JWT payload claim.
         *
         * @link https://tools.ietf.org/html/rfc7519#section-4.1.4
         *
         * @return bool|string
         */
        protected static function getExpClaim()
        {
            return self::getPayloadClaim(JWT::EXPIRY);
        }

        /**
         * Check the validity of the JWT's expiration claim as defined in the
         * token payload. Returns false if the expiration time has surpassed the
         * current time.
         *
         * @param int $expiry
         *
         * @return bool
         * @throws InvalidJwtException
         * @throws TokenExpiredException
         */
        protected static function validateExpClaim($expiry)
        {
            if (is_numeric($expiry) && (string)$expiry == (int)$expiry) {
                if ( (int)$expiry < (time() - self::$leeway)) {
                    throw new TokenExpiredException('The JWT is no longer valid.');
                }
            } else {
                throw new InvalidJwtException('Invalid "exp" value.');
            }

            return true;
        }

        // NBF: Not Before
        /**
         * Set the "Not Before" [nbf] JWT payload claim.
         * This sets the time after which the JWT can be accepted.
         *
         * This method will be [re]set the IAT value if IAT is AFTER
         * the time defined for NBF
         *
         * This expects UNIX TIMESTAMP[int] value
         *
         * @link https://tools.ietf.org/html/rfc7519#section-4.1.5
         *
         * @param int|bool $nbf UNIX TIMESTAMP
         */
        final public static function setNbfClaim($nbf = false)
        {
            if( ($nbf === false) || (!is_int($nbf)) ) {
                /** @var int $notBefore */
                $nbf = time();
            }

            self::setPayloadClaim(JWT::NOT_BEFORE, $nbf);
            self::setPayloadClaim(JWT::ISSUED_AT, $nbf);

        }

        /**
         * Retrieves the "Not Before" [nbf] JWT payload claim.
         *
         * @link https://tools.ietf.org/html/rfc7519#section-4.1.5
         *
         * @param string|bool $nbf
         *
         * @return bool|string
         */
        protected static function getNbfClaim()
        {
            return self::getPayloadClaim(JWT::NOT_BEFORE);
        }

        /**
         * Validate the "nbf" (not before) claim, if included.
         *
         * @param int $nbf
         *
         * @return bool
         * @throws BeforeValidException
         * @throws InvalidJwtException
         */
        protected static function validateNbfClaim($nbf)
        {
            if (is_numeric($nbf) && (string)$nbf == (int)$nbf) {
                if ( (int)$nbf > time() ) {
                    throw new BeforeValidException('The JWT time stamp not valid.');
                }
            } else {
                throw new InvalidJwtException('Invalid "nbf" value.');
            }

            return true;
        }

        // IAT: Issued At
        /**
         * Set the "Issued At" [iat] JWT payload claim.
         * This sets the time at which the JWT was issued / created.
         *
         * This method will be [re]set the NBF value to the given IAT value
         * if NBF is defined to a time BEFORE for IAT
         *
         * This expects UNIX TIMESTAMP[int] value
         *
         * @link https://tools.ietf.org/html/rfc7519#section-4.1.6
         *
         * @param int|bool $iat UNIX TIMESTAMP
         *
         * @return JWT
         */
        public static function setIatClaim($iat = false)
        {
            if( ($iat === false) || (!is_int($iat)) ) {
                if ( (self::getIatClaim() !== false) ) {
                    $iat = self::getIatClaim();
                } else  {
                    $iat = time();
                }

            }

            // make sure NBF is at or after IAT
            if (self::getNbfClaim() < $iat) {
                self::setNbfClaim($iat);
            }

            /** @TODO Add UNIX TIMESTAMP validation */
            self::setPayloadClaim(JWT::ISSUED_AT, $iat);

        }

        /**
         * Retrieves the "Issued At" [iat] JWT payload claim.
         *
         * @link https://tools.ietf.org/html/rfc7519#section-4.1.6
         *
         * @return bool|int
         */
        protected static function getIatClaim()
        {
            return self::getPayloadClaim(JWT::ISSUED_AT);
        }

        /**
         * Validate the "iat" (issued at) claim.
         *
         * @param int $iat UNIX TIMESTAMP
         *
         * @return bool
         * @throws BeforeValidException
         * @throws InvalidJwtException
         */
        protected static function validateIatClaim($iat)
        {
            if (is_numeric($iat) && (string)$iat == (int)$iat) {
                if ( (int)$iat > time() ) {
                    throw new BeforeValidException('The JWT time stamp not valid.');
                }
            } else {
                throw new InvalidJwtException('Invalid "iat" value.');
            }

            return true;
        }

        // JTI: JWT Token Identifier
        /**
         * Set the JWT Token Identifier [jti] payload claim automatically.
         * This defines a unique identifier for each token.
         *
         * @link https://tools.ietf.org/html/rfc7519#section-4.1.7
         *
         * @param bool $value used for validation purposes
         *
         * @return JWT|tClaims
         */
        protected function setJwtId($value = false)
        {
            if(($value === false) || ($value === null)) {
                /** @TODO convert to use Libsodium when in PHP 7 */
                $value = B64Encoder::encode(mcrypt_create_iv(32));
            }

            $this->setPayloadClaim(JWT::JID, $value);

            return $this;
        }

        /**
         * Retrieves the "Token Identifier" [jti] JWT payload claim.
         *
         * @link https://tools.ietf.org/html/rfc7519#section-4.1.7
         *
         * @return bool|int UNIX TIMESTAMP
         */
        protected function getJwtId()
        {
            return $this->getPayloadClaim(JWT::ISSUED_AT);
        }

        // Data Claim
        /**
         * Set the DATA segment payload claim; e: Private Claim Names
         *
         * @link https://tools.ietf.org/html/rfc7519#section-4.3
         *
         * @param array|string $payload
         *
         * @return JWT|tClaims
         */
        final public function setDataClaim($payload)
        {
            $this->setPayloadClaim('data', $payload);

            return $this;
        }

        /**
         * Retrieves the "Claims Data" array from a given Token
         *
         * @param $tokenString
         *
         * @return array|bol
         * @throws UndefinedTokenException
         */
        final static function getDataClaim($tokenString) {
            if ( (strtolower($tokenString) === 'null')
                || (!isset($tokenString[0]))
            ) {
                throw new UndefinedTokenException();
            }

            /** @var object $tokenObject */
            $tokenObject = self ::validateStructure($tokenString);

            return self::decodeClaims($tokenObject->claims);
        }

    }