<?php

    namespace DPP\Security\Cryptography\JWT;

    use DPP\Security\Cryptography\JWT\Exceptions\InvalidJwtException;
    use DPP\Security\Cryptography\JWT\Exceptions\JwtException;
    use DPP\Security\Cryptography\JWT\Exceptions\UnsupportedAlgorithmException;
    use DPP\Security\Encoders\B64Encoder;
    use DPP\Security\Encoders\JsonEncoder;

    trait tGenerate
    {

        use tHeader;
        use tClaims;

        /** @var string $algorithm */
        private static $algorithm;

        /**
         * The secret string for encoding the JWT signature.
         *
         * @var string
         */
        private static $secret;

        /**
         * The header data included with the JWT.
         *
         * @link https://tools.ietf.org/html/rfc7519#section-5
         *
         * @var array
         */
        private static $header;
        private static $headerJson;

        /** @var int Grace period in seconds to allow for clock skew. Defaults to 0 seconds. */
        private static $leeway = 2;

        // *******************************************************************
        // Class methods

        /**
         * Returns a JWT encoded value
         *
         * @return string
         * @throws InvalidJwtException
         * @throws JwtException
         * @throws \DPP\Security\Encoders\Exceptions\JsonEncodingException
         */
        private static function generate() {

            // Issued At Time: time when the token was generated
            // This can be altered using JWT::setPayloadClaim('iat', [your UNIX TIMESTAMP])
            // But it really should be set automatically
            self::setIatClaim();

            // Token not valid after given time
            self::setExpClaim();

            return self::encodeHeaders() . '.'
                 . self::encodeClaims()  . '.'
                 . self::encodeSignature();
        }

        // Header management methods ************************************

        /**
         * Set Header definition for Token validation.
         *
         * @param object $headerObject
         *
         * @return void
         */
        private static function setHeaders ($headerObject)
        {
            self::$header = $headerObject;
        }

        /**
         * Retrieve the current Header definition.
         *
         * @return array
         */
        private static function getHeaders ()
        {
            // Header must be an array per RFC
            return (array)self::$header;
        }

        /**
         * JSON Encode Header Object
         *
         * @return
         * @throws JsonEncodingException
         */
        private static function encodeHeaders() {
            self::$headerJson = JsonEncoder::B64URLsafeEncode(self::getHeaders());

            return self::getEncodedHeaders();
        }

        /**
         * Return JSON Encoded Header Object
         */
        private static function getEncodedHeaders() {
            return self::$headerJson;
        }


        // Claim management methods *************************************

        /**
         * Set Claims definition for Token validation.
         *
         * @param object $claimsObject
         *
         * @return void
         */
        private static function setClaims ($claimsObject)
        {
            // Claims must be an array per RFC
            self::$claims = $claimsObject;
        }

        /**
         * Retrieve the current claim definition.
         *
         * @return object
         */
        private static function getClaims ()
        {
            return self::$claims;
        }

        /**
         * JSON Encode Claims Object
         */
        private static function encodeClaims() {
            self::$claimsJson = JsonEncoder::B64URLsafeEncode(self::getClaims());

            return self::getEncodedClaims();
        }

        /**
         * Return JSON Encoded Claims Object
         */
        private static function getEncodedClaims() {
            return self::$claimsJson;
        }

        // Signature management methods *********************************

        /**
         * Generate the JWT signature. The header and payload are encoded,
         * concatenated with a dot, hashed via sha256 with a secret, and then
         * encoded and returned.
         *
         * @return string
         * @throws JwtException
         */
        private static function encodeSignature()
        {
            $method = 'encodeWith' . self::getAlgorithm();

            if (!method_exists(__CLASS__, $method)) {
                throw new UnsupportedAlgorithmException(sprintf('Unsupported algorithm ("%s").', self::$algorithm));
            }

            $encoded = self::$method(
                self::getSecretKey(),
                self::getEncodedHeaders() . "." . self::getEncodedClaims()
            );

            return B64Encoder::URLsafeEncode($encoded);
        }

        /**
         * Retrieve the type of Algorithm to be used to encrypt token.
         *
         * @link https://tools.ietf.org/html/rfc7519#section-3.1
         *
         * @return string|bool $algTyped
         */
        private static function getAlgorithm()
        {
            return self::$header->alg;
        }


        // Encryption management methods ********************************

        /**
         * Define the secret key to use for encryption.
         *
         * @param  string $keyValue
         *
         * @return Generate
         */
        public static function setSecretKey($keyValue)
        {
            self::$secret = $keyValue;

            return __CLASS__;
        }

        /**
         * Retrieve the secret key to use for encryption.
         *
         * @return string
         */
        private static function getSecretKey()
        {
            return self::$secret;
        }


        // MISC methods *********************************************

        /**
         * Define time "leeway".
         *
         * @link https://tools.ietf.org/html/rfc7519#section-4.1.3
         *
         * @param $value
         *
         * @return JWT
         */
        public function setLeeway($value = false)
        {
            if( ($value === false) || (!is_int($value)) ) {
                $value = self::DEFAULT_LEEWAY;
            }

            $this->leeway = $value;

            return $this;
        }

        /**
         * Define time "leeway".
         *
         * @link https://tools.ietf.org/html/rfc7519#section-4.1.3
         *
         * @return int
         */
        private static function getLeeway()
        {
            // Safety check. If property is not defined [somehow],
            // set it to default value.
            if( (self::$leeway === false) || (!is_int(self::$leeway)) ) {
                self::$leeway = DEFAULT_LEEWAY;
            }

            return self::$leeway;
        }

    }
