<?php

    namespace DPP\Security\Cryptography\JWT;


    use DPP\Security\Cryptography\JWT\Exceptions\BeforeValidException;
    use DPP\Security\Cryptography\JWT\Exceptions\InvalidJwtException;
    use DPP\Security\Cryptography\JWT\Exceptions\TokenExpiredException;
    use DPP\Security\Encoders\B64Encoder;

    trait tClaims
    {

        /**
         * Holds the JWT payload claims.
         *
         * @link https://tools.ietf.org/html/rfc7519#section-4.1
         *
         * @var object
         */
        private static $claims;
        private static $claimsJson;

        /**
         * Defines the expiry date/time
         * 15 minutes from current time
         *
         * @link https://tools.ietf.org/html/rfc7519#section-4.1.3
         *
         * @var string
         */
        private static $expiry;

        /** Default value, in minutes */
        private static $default_expiry = 15;

// *******************************************************************
// Class methods

        /**
         * Set a custom payload claim on the JWT.
         * The RFC calls these private claims.
         * Eg you may wish to set a user_id or a username in the JWT payload
         *
         * @param string $claimName
         * @param mixed  $claimValue
         *
         * @return void
         */
        protected static function setPayloadClaim ($claimName, $claimValue)
        {
            $claimName = strtolower($claimName);
            self::$claims->$claimName = $claimValue;
        }

        /**
         * Retrieves payload claim on the JWT.
         *
         * @param string $claimName
         *
         * @return string|boolean
         */
        protected static function getPayloadClaim ($claimName)
        {
            if (array_key_exists('iat', self::$claims)) {
                $claimValue = self::$claims->$claimName;
            } else {
                $claimValue = false;
            }

            return $claimValue;
        }

        /**
         * remove a claim from current definition.
         *
         * @param string $claimName
         *
         * @return void
         */
        protected function unsetPayloadClaim ($claimName)
        {
            unset(self::$claims[$claimName]);
        }

        /**
         * remove entire claims definition.
         *
         * @return void
         */
        protected function clearPayloadClaims ()
        {
            self::$claims = (object)[ 'iss'  => false  // ISSuer
                , 'sub'  => false  // Subject
                , 'aud'  => false  // Audience
                , 'exp'  => false  // EXPires
                , 'nbf'  => false  // Not BeFore
                , 'iat'  => false  // Issued At Time: time when the token was generated
                , 'jti'  => false  // Json Token Id: an unique identifier for the token
                , 'data' => false  // DATA related to the signer user
            ];
        }

        // ISS: Issuer
        /**
         * Set the Issuer [iss] JWT payload claim.
         * This defines who issued the token.
         *
         * Can be a string or URI.
         *
         * @link https://tools.ietf.org/html/rfc7519#section-4.1.1
         *
         * @param string|bool $iss
         *
         * @return JWT
         */
        final public function setIssClaim($iss = false)
        {
            if($iss === false) {
                /** @var string $iss */
                $iss = (@$_SERVER['SERVER_NAME'] ?: 'cli');
            }

            $this->setPayloadClaim('iss', $iss);

            return $this;
        }

        /**
         * Retrieve the Issuer [iss] JWT payload claim.
         *
         * Can be a string or URI.
         *
         * @link https://tools.ietf.org/html/rfc7519#section-4.1.1
         *
         * @return bool|string
         */
        protected function getIssClaim()
        {
            return $this->getPayloadClaim('iss');
        }

        /**
         * Validate the "iss" (issuer) claim.
         *
         * @return bool
         * @throws InvalidClaimTypeException
         */
        protected function validateIssClaim()
        {
            $iss = $this->getPayloadClaim ('iss');

            if ($iss) {
                Validation::checkClaimType('iss', $iss, 'string');
            }

            return true;
        }

        // SUB: Subject
        /**
         * Set the Subject [sub] JWT payload claim.
         * This defines the principal subject for the token.
         *
         * Can be a string or URI.
         *
         * @link https://tools.ietf.org/html/rfc7519#section-4.1.2
         *
         * @param string|bool $sub
         *
         * @return JWT
         */
        final public function setSubClaim($sub = false)
        {
            $this->setPayloadClaim('sub', $sub);

            return $this;
        }

        /**
         * Retrieve the Subject [sub] JWT payload claim.
         *
         * Can be a string or URI.
         *
         * @link https://tools.ietf.org/html/rfc7519#section-4.1.2
         *
         * @param string|bool $sub
         *
         * @return bool|string
         */
        protected function getSubClaim()
        {
            return $this->getPayloadClaim('sub');
        }

        // AUD: Audience
        /**
         * Set the Audience [aud] JWT payload claim.
         * This defines to whom the token is issued for.
         *
         * Can be a string or URI.
         *
         * @link https://tools.ietf.org/html/rfc7519#section-4.1.3
         *
         * @param string|bool $aud
         *
         * @return JWT
         */
        final public function setAudClaim($aud = false)
        {
            $this->setPayloadClaim('aud', $aud);

            return $this;
        }

        /**
         * Retrieves the Audience [aud] JWT payload claim.
         *
         * @link https://tools.ietf.org/html/rfc7519#section-4.1.3
         *
         * @param string|bool $sub
         *
         * @return bool|string
         */
        protected function getAudClaim()
        {
            return $this->getPayloadClaim('aud');
        }

        /**
         * Validate the "aud" (audience) claim, if included.
         *
         * @param  string|string[] $audience
         * @return void
         * @throws InvalidJwtException
         */
        protected function validateAudClaim($audience)
        {
            if (array_key_exists('aud', self::$claims)) {
                $audience = is_array($audience) ? $audience : [$audience];

                // Make sure the audience claim is set to a valid value
                $foundAudience = is_array(self::$claims['aud']) ? self::$claims['aud'] : [self::$claims['aud']];
                $expectedIndex = 0;

                foreach ($foundAudience as $index => $value) {
                    if (is_string($value) && $index === $expectedIndex) {
                        $expectedIndex++;
                    } else {
                        throw new InvalidJwtException('Invalid "aud" value.', $this);
                    }
                }

                // Make sure the JWT is intended for any of the expected audiences
                foreach ($audience as $expectedAudience) {
                    if (in_array($expectedAudience, $foundAudience, true)) {
                        return;
                    }
                }

                throw new InvalidJwtException('Invalid JWT audience.', $this);
            }
        }

        // EXP: Expiration Time
        /**
         * Set the Expiration [exp] JWT payload claim.
         * This sets the time at which the JWT token will expire and
         * is no longer be accepted.
         *
         * This expects UNIX TIMESTAMP[int] value
         *
         * @link https://tools.ietf.org/html/rfc7519#section-4.1.4
         *
         * @param int $exp
         *
         * @return JWT
         * @throws \Exception
         */
        protected static function setExpClaim($exp = false)
        {
            /** @var int  $expiry token EXPIRY value */
            $expiry = self::getExpiry();

            // If not given, see what we have already
            if(($exp === false) || (!is_int($exp))) {
                $exp = self::getExpClaim();
            }

            // If still nothing, Set EXP based off IAT
            if($exp === false) {
                // Pull current "Issued At Time"
                $iat = self::getPayloadClaim('iat');

                $date = new \DateTime(strtotime($iat));
                $date->modify("+" . $expiry . " minutes"); //or whatever value you want

                /** @var int $expiration */
                $exp = (int)$date->format('U');
            }

            self::setPayloadClaim('exp', $exp);
        }

        /**
         * Retrieves the Expiration [exp] JWT payload claim.
         *
         * @link https://tools.ietf.org/html/rfc7519#section-4.1.4
         *
         * @param string|bool $sub
         *
         * @return bool|string
         */
        protected static function getExpClaim()
        {
            return self::getPayloadClaim('exp');
        }

        /**
         * Check the validity of the JWT's expiration claim as defined in the
         * token payload. Returns false if the expiration time has surpassed the
         * current time.
         *
         * @param int $exp
         *
         * @return bool
         * @throws InvalidJwtException
         * @throws TokenExpiredException
         */
        protected static function validateExpClaim($exp)
        {
            if (is_numeric($exp) && (string)$exp == (int)$exp) {
                if ((time() - self::$leeway) >= (int)$exp) {
                    throw new TokenExpiredException('The JWT is no longer valid.');
                }
            } else {
                throw new InvalidJwtException('Invalid "exp" value.');
            }

            return true;
        }

        // NBF: Not Before
        /**
         * Set the "Not Before" [nbf] JWT payload claim.
         * This sets the time after which the JWT can be accepted.
         *
         * This method will be [re]set the IAT value if IAT is AFTER
         * the time defined for NBF
         *
         * This expects UNIX TIMESTAMP[int] value
         *
         * @link https://tools.ietf.org/html/rfc7519#section-4.1.5
         *
         * @param int|bool $nbf UNIX TIMESTAMP
         *
         * @return JWT
         */
        final public static function setNbfClaim($nbf = false)
        {
            if( ($nbf === false) || (!is_int($nbf)) ) {
                /** @var int $notBefore */
                $nbf = time();
            }

            self::setPayloadClaim('nbf', $nbf);
            self::setPayloadClaim('iat', $nbf);

        }

        /**
         * Retrieves the "Not Before" [nbf] JWT payload claim.
         *
         * @link https://tools.ietf.org/html/rfc7519#section-4.1.5
         *
         * @param string|bool $nbf
         *
         * @return bool|string
         */
        protected static function getNbfClaim()
        {
            return self::getPayloadClaim('nbf');
        }

        /**
         * Validate the "nbf" (not before) claim, if included.
         *
         * @param int $nbf
         *
         * @return bool
         * @throws BeforeValidException
         * @throws InvalidJwtException
         */
        protected static function validateNbfClaim($nbf)
        {
            if (is_numeric($nbf) && (string)$nbf == (int)$nbf) {
                if ((time() - self::$leeway) < (int)$nbf) {
                    throw new BeforeValidException('The JWT time stamp not valid.');
                }
            } else {
                throw new InvalidJwtException('Invalid "nbf" value.');
            }

            return true;
        }

        // IAT: Issued At
        /**
         * Set the "Issued At" [iat] JWT payload claim.
         * This sets the time at which the JWT was issued / created.
         *
         * This method will be [re]set the NBF value to the given IAT value
         * if NBF is defined to a time BEFORE for IAT
         *
         * This expects UNIX TIMESTAMP[int] value
         *
         * @link https://tools.ietf.org/html/rfc7519#section-4.1.6
         *
         * @param int|bool $iat UNIX TIMESTAMP
         *
         * @return JWT
         */
        public static function setIatClaim($iat = false)
        {
            if( ($iat === false) || (!is_int($iat)) ) {
                if ( (self::getIatClaim() !== false) ) {
                    $iat = self::getIatClaim();
                } else  {
                    $iat = time();
                }

            }

            // make sure NBF is at or after IAT
            if (self::getNbfClaim() < $iat) {
                self::setNbfClaim($iat);
            }

            /** @TODO Add UNIX TIMESTAMP validation */
            self::setPayloadClaim('iat', $iat);

        }

        /**
         * Retrieves the "Issued At" [iat] JWT payload claim.
         *
         * @link https://tools.ietf.org/html/rfc7519#section-4.1.6
         *
         * @param string|bool $iat
         *
         * @return bool|string
         */
        protected static function getIatClaim()
        {
            return self::getPayloadClaim('iat');
        }

        /**
         * Validate the "iat" (issued at) claim.
         *
         * @param int $iat
         *
         * @return bool
         * @throws BeforeValidException
         * @throws InvalidJwtException
         */
        protected static function validateIatClaim($iat)
        {
            if (is_numeric($iat) && (string)$iat == (int)$iat) {
                if ((time() - self::$leeway) < (int)$iat) {
                    throw new BeforeValidException('The JWT time stamp not valid.');
                }
            } else {
                throw new InvalidJwtException('Invalid "iat" value.');
            }

            return true;
        }

        // JTI: JWT Token Identifier
        /**
         * Set the JWT Token Identifier [jti] payload claim automatically.
         * This defines a unique identifier for each token.
         *
         * @link https://tools.ietf.org/html/rfc7519#section-4.1.7
         *
         * @param bool $value used for validation purposes
         *
         * @return JWT
         */
        protected function setJwtId($value = false)
        {
            if(($value === false) || ($value === null)) {
                /** @TODO convert to use 'random_bytes' when in PHP 7 */
                $value = B64Encoder::encode(mcrypt_create_iv(32));
            }

            $this->setPayloadClaim('jti', $value);

            return $this;
        }

        /**
         * Retrieves the "Token Identifier" [jti] JWT payload claim.
         *
         * @link https://tools.ietf.org/html/rfc7519#section-4.1.7
         *
         * @param string|bool $iat
         *
         * @return bool|string
         */
        protected function getJwtId()
        {
            return $this->getPayloadClaim('iat');
        }

        // Data Claim
        /**
         * Set the DATA segment payload claim; e: Private Claim Names
         *
         * @link https://tools.ietf.org/html/rfc7519#section-4.3
         *
         * @param array|string $payload
         *
         * @return JWT
         */
        final public function setDataClaim($payload)
        {
            $this->setPayloadClaim('data', $payload);

            return $this;
        }

        // MISC methods *********************************************

        /**
         * Set EXPIRY value
         *
         * @param int $exp
         *
         * @return JWT
         */
        public function setExpiry($exp = false) {

            if( ($exp === false) || (!is_int($exp)) ) {
                /** @var int $exp */
                $exp = self::$default_expiry;
            }

            self::$expiry = $exp;

            return $this;
        }

        /**
         * retrieve Token EXPIRY
         *
         * @return int
         */
        protected static function getExpiry() {

            if( (self::$expiry === false)
             || (self::$expiry === null)
             || (!is_int(self::$expiry))
                ) {
                self::$expiry = self::$default_expiry;
            }

            return self::$expiry;
        }

    }