<?php

    namespace DPP\Security\Cryptography\JWT;

    use \DateTime;

    use DPP\Security\Cryptography\Algorithm\HMAC\HS256;
    use DPP\Security\Cryptography\JWT\Exceptions\InvalidClaimsStructureException;
    use DPP\Security\Cryptography\JWT\Exceptions\InvalidClaimTypeException;
    use DPP\Security\Cryptography\JWT\Exceptions\InvalidHeaderStructureException;
    use DPP\Security\Cryptography\JWT\Exceptions\InvalidStructureException;
    use DPP\Security\Cryptography\JWT\Exceptions\InvalidTokenTypeException;
    use DPP\Security\Cryptography\JWT\Exceptions\TokenExpiredException;
    use DPP\Security\Cryptography\JWT\Exceptions\TokenInactiveException;
    use DPP\Security\Cryptography\JWT\Exceptions\UndefinedAlgorithmException;
    use DPP\Security\Cryptography\JWT\Exceptions\UndefinedEncryptionKeyException;
    use DPP\Security\Cryptography\JWT\Exceptions\UndefinedTokenException;
    use DPP\Security\Cryptography\JWT\Exceptions\UnsecureTokenException;
    use DPP\Security\Cryptography\JWT\Exceptions\UnsupportedAlgorithmException;
    use DPP\Security\Cryptography\JWT\Exceptions\UnsupportedTokenTypeException;
    use DPP\Security\Encoders\Exceptions\JsonDecodingException;
    use DPP\Security\Encoders\JsonEncoder;


    /**
     * This class contains methods used for validating JWT tokens.

     */
    class Validate
    {

        use tGenerate;
        use HS256;

        /** @TODO move this to a Algorithm Class */
        const ALGORITHM_HS256 = 'HS256';

        /**
         * Default algorithm to use when encoding token when algorithm is
         * provided in token header nor as parameter to encode method.
         */
        const DEFAULT_ALGORITHM = self::ALGORITHM_HS256;

        /**
         * Default "type" of JWT to be created, usually just JWT.
         *
         * @link https://tools.ietf.org/html/rfc7519#section-5.1
         *
         * @var string
         */
        const DEFAULT_TYPE = 'JWT';


        // *******************************************************************
        // Class methods

        public static function token($tokenString, $key)
        {
            if (!isset($tokenString[0])) {
                throw new UndefinedTokenException();
            }

            if (!isset($key[0])) {
                throw new UndefinedEncryptionKeyException();
            }

            self ::setSecretKey($key);

            /** @var object $tokenObject */
            $tokenObject = self ::validateStructure($tokenString);

            // Decode, validate and set Header
            self ::validateHeader($tokenObject -> header);

            // Decode, validate and set Claims
            self ::validateClaims($tokenObject -> claims);

            // Generate new token
            $newToken = self ::generate();

            // [re]Encode a Signature to verify against given token
            return self ::validateSignature($tokenObject -> signature);
        }

        /**
         * Confirm the structure of a JSON Web Token, it has three parts
         * separated by dots and complies. THis will return the token separated
         * into its constituent parts as an object
         *
         * @param string $encodedToken
         *
         * @return object
         * @throws InvalidStructureException
         */
        protected static function validateStructure($encodedToken)
        {
            $encodedSegments = explode('.', $encodedToken);

            if (count($encodedSegments) !== 3) {
                throw new InvalidStructureException('Unexpected number of JWT segments [' . count($encodedSegments) . ']');
            }

            return (object)[
                'header' => $encodedSegments[0]
                ,
                'claims' => $encodedSegments[1]
                ,
                'signature' => $encodedSegments[2]
            ];
        }

        /**
         * validates the header unto itself and returns it as an Object
         *
         * @param string $encodedHeader
         *
         * @return void
         * @throws InvalidHeaderStructureException
         * @throws InvalidTokenTypeException
         * @throws JsonDecodingException
         * @throws UndefinedAlgorithmException
         */
        protected static function validateHeader($encodedHeader)
        {
            /** @var array $decodedHeader */
            $decodedHeader = JsonEncoder ::B64URLsafeDecode($encodedHeader);

            if (!is_array($decodedHeader)) {
                throw new InvalidHeaderStructureException('Invalid Header Structure');
            }

            // Pull out defined encryption type, if there is one, and make sure we can use it.
            if (array_key_exists('alg', $decodedHeader)) {
                self::$algorithm = $decodedHeader['alg'];
                /** @TODO validate algorithm type can be used */
            } else {
                throw new UndefinedAlgorithmException('Undefined Algorithm');
            }

            // Verify JWT Type is proper
            if ((array_key_exists('typ', $decodedHeader))
                && (strtolower($decodedHeader['typ']) !== 'jwt')
            ) {
                throw new InvalidTokenTypeException('Invalid Header Structure');
            }

            self::setHeaders((object)$decodedHeader);
        }

        /**
         * validates Claims unto itself and returns it as an Object
         *
         * @param string $encodedClaims
         *
         * @return void
         * @throws Exceptions\BeforeValidException
         * @throws Exceptions\InvalidJwtException
         * @throws InvalidClaimsStructureException
         * @throws JsonDecodingException
         * @throws TokenExpiredException
         */
        protected static function validateClaims($encodedClaims)
        {
            /** @var array $decodedClaims */
            $decodedClaims = JsonEncoder ::B64URLsafeDecode($encodedClaims);

            if (!is_array($decodedClaims)) {
                throw new InvalidClaimsStructureException('Invalid Claims Structure');
            }

            // Pull out Claim types and validate.
            self ::validateTimestamps($decodedClaims);

            self ::setClaims((object)$decodedClaims);
        }

        /**
         * validates Signature unto itself and returns a boolean
         * indicating PASS or FAIL
         *
         * @param string $encodedSignature
         *
         * @return bool
         * @throws Exceptions\JwtException
         */
        protected static function validateSignature($encodedSignature)
        {
            return ($encodedSignature === self ::encodeSignature());
        }

        /**
         * Verifies if timestamp claims like iat, exp, nbf are invalid.
         *
         * @param $decodedClaims
         *
         * @throws Exceptions\BeforeValidException
         * @throws Exceptions\InvalidJwtException
         * @throws TokenExpiredException
         */
        protected static function validateTimestamps($decodedClaims)
        {
            $decodedClaimsObject = (object)$decodedClaims;

            /** @var int $leeway */
            $leeway = self::getLeeway();

            /** @var DateTime $timestamp */
            $timestamp = (strtotime("-" .$leeway . " seconds"));

            // Is this token expired
            self::validateExpClaim($decodedClaims['exp']);

            // Was this issued AFTER current time
            self::validateIatClaim($decodedClaims['iat']);

            // Are we trying to use this BEFORE we are allowed
            self::validateIatClaim($decodedClaims['nbf']);
        }
    }

