<?php

    namespace DPP\Security\Cryptography\Algorithm\HMAC;

    use DPP\Security\Cryptography\Algorithm\Exceptions\InvalidKeyException;

    /**
     * A trait for enabling support of the "HS256" algorithm.
     */
    trait HS256
    {
        /**
         * Encode given data using a given key.
         *
         * @param  mixed  $key
         * @param  string $dataToSign
         * @return string|null
         * @throws InvalidKeyException
         */
        protected static function encodeWithHS256($key, $dataToSign)
        {
            if (!is_string($key)) {
                throw new InvalidKeyException('Invalid key.');
            }

            return hash_hmac('SHA256'
                            , $dataToSign
                            , $key
                            , true
                            );
        }

        /**
         * Verify a given signature using a given key.
         *
         * @param  mixed  $key
         * @param  string $dataToSign
         * @param  string $signature
         * @return bool
         * @throws InvalidKeyException
         */
        protected static function verifyWithHS256($key, $dataToSign, $signature)
        {
            if (!is_string($key)) {
                throw new InvalidKeyException('Invalid key.');
            }

            return hash_equals($signature, self::encodeWithHS256($key, $dataToSign));
        }
    }