<?php

namespace DPP\Financial\Scheduling;

use DPP\Financial\Scheduling\Factorization\FactorizationScheduler;
use DPP\Financial\Scheduling\Amortization\NormalAmortizationScheduler;
use DPP\Financial\FinancialException;

/**
 * Class Method
 *
 * @package DPP\Financial\Amortization
 */
class Method
{
    private $name;
    private $short_name;

    private $class_name;
    /**
     * Method constructor.
     *
     * @param string $n Name
     * @param string $sn Short name
     * @param string $cn Class name
     */
    public function __construct($n, $sn, $cn)
    {
        $this->name       = $n;
        $this->short_name = $sn;
        $this->class_name = $cn;
    }

    /** Static initializer for Normal Amortization
     *
     * @return Method
     */
    final public static function NORMAL()
    {
        return new Method(NormalAmortizationScheduler::NAME, NormalAmortizationScheduler::SHORT_NAME, NormalAmortizationScheduler::CLASS_NAME);
    }

    /** Static initializer for Fixed Principal Amortization
     *
     * @return Method
     */
    final public static function FIXED_PRINCIPAL()
    {
        return new Method(FixedPrincipalAmortizationCalculator::NAME, FixedPrincipalAmortizationCalculator::SHORT_NAME, FixedPrincipalAmortizationCalculator::CLASS_NAME);
    }

    /** Static initializer for Interest Only Amortization
     *
     * @return Method
     */
    final public static function INTEREST_ONLY()
    {
        return new Method(InterestOnlyAmortizationCalculator::NAME, InterestOnlyAmortizationCalculator::SHORT_NAME, InterestOnlyAmortizationCalculator::CLASS_NAME);
    }

    final public static function FACTORIZATION()
    {
        return new Method(FactorizationScheduler::NAME, FactorizationScheduler::SHORT_NAME, FactorizationScheduler::CLASS_NAME);
    }

    /** Gets a list of available Methods
     *
     * @return self[]
     */
    final public static function getList()
    {
        $rt = array();

        $rt[] = self::NORMAL();
        $rt[] = self::FIXED_PRINCIPAL();
        $rt[] = self::INTEREST_ONLY();
        $rt[] = self::FACTORIZATION();

        return $rt;
    }

    /**
     * @param string $ms The string description of the method desired
     * @return Method The final method object
     * @throws FinancialException
     */
    final public static function getMethod($ms)
    {
        $rt  = null;
        $mtd = self::getList();

        foreach ($mtd as $m)
        {
            if ($m->shortName() === $ms)
            {
                $rt = $m;
                break;
            }
        }

        if ($rt === null)
        {
            throw new FinancialException(sprintf("Method string '%s' is invalid", $ms));
        }

        return $rt;
    }

    /**
     * @return string
     */
    public function name()
    {
        return $this->name;
    }

    /**
     * @return string
     */
    public function shortName()
    {
        return $this->short_name;
    }

    /**
     * @return string
     */
    public function className()
    {
        return $this->class_name;
    }
}
