<?php

namespace DPP\Financial\Time;

use \DateTime;
use \DateInterval;

use DPP\Financial\FinancialException;

/**
 * Class SemiMonthly
 *
 * @package DPP\Financial\Time
 */
class SemiMonthly extends Frequency
{
    const DEFAULT_FIRST = 1;
    const DEFAULT_LAST = 15;

    private $firstDate;
    private $lastDate;

    /**
     * SemiMonthly constructor.
     *
     * @param DateTime|null $fd
     * @param DateTime|null $ld
     */
    public function __construct(DateTime $fd = null, DateTime $ld = null)
    {
        parent::__construct("Semi Monthly", "semi_monthly");

        $this->firstDate = $fd;
        $this->lastDate = $ld;
    }

    /**
     * @return float|int
     */
    public function partOfYear()
    {
        return ($this->lastDateInterval !== null) ? ($this->lastDateInterval->format("%a") / 365) : 0;
    }

    /**
     * @param DateTime $dt
     * @return DateTime
     * @throws FinancialException
     */
    public function getNextDate(DateTime $dt)
    {
        $rt = $this->copy($dt);
        $dim = cal_days_in_month(CAL_GREGORIAN, (int) $dt->format("m"), (int) $dt->format("Y"));

        $fdd = ($this->firstDate !== null) ? (int) $this->firstDate->format("d") : self::DEFAULT_FIRST;
        $ldd = ($this->lastDate !== null) ? (int) $this->lastDate->format("d") : self::DEFAULT_LAST;
        $ldd = ($ldd > $dim) ? $dim : $ldd;

        $pdt = (int) $dt->format("d");

        if (($pdt >= $fdd) && ($pdt < $ldd))
        {
            $rt->setDate((int) $dt->format("Y"), (int) $dt->format("m"), $ldd);
        }
        else
        {
            $rt->setDate((int) $rt->format("Y"), (int) $rt->format("m"), 1);

            $rt->add(DateInterval::createFromDateString("1 month"));
            $rt->setDate((int) $rt->format("Y"), (int) $rt->format("m"), $fdd);
        }

        $this->lastDateInterval = $rt->diff($dt);

        return $rt;
    }

    /**
     * @return int
     */
    public function termsInAYear()
    {
        return 24;
    }

    /**
     * @return int
     */
    public function daysInTerm()
    {
        return 15;
    }
}
