<?php

namespace DPP\Financial;

use \DateTime;

use DPP\Financial\Time\Frequency;

/**
 * Class ScheduledPayment
 *
 * @package DPP\Financial
 */
class ScheduledPayment
{
    private $payment_date;
    private $period_start_date;
    private $annual_interest_rate;
    private $payment_amount;
    private $total_amount_after_payment;
    private $interest_amount_paid;
    private $principal_amount_paid;
    private $locked = false;

    /**
     * ScheduledPayment constructor.
     *
     * @param DateTime $pd
     * @param DateTime $psd
     * @param float    $air
     * @param null     $pa
     */
    public function __construct(DateTime $pd, DateTime $psd, $air, $pa = null)
    {
        $this->payment_date         = $pd;
        $this->period_start_date    = $psd;
        $this->annual_interest_rate = $air;
        $this->payment_amount       = $pa;
    }

    /**
     * @return DateTime
     */
    public function date()
    {
        return $this->payment_date;
    }

    /**
     * @return float
     */
    public function part()
    {
        return $this->length() / 365;
    }

    /**
     * @return float
     */
    public function principal()
    {
        return $this->principal_amount_paid;
    }

    /**
     * @return float
     */
    public function interest()
    {
        return $this->interest_amount_paid;
    }

    /**
     * @return float
     */
    public function balance()
    {
        return $this->total_amount_after_payment;
    }

    /**
     * @return float
     */
    public function amount()
    {
        return $this->principal_amount_paid + $this->interest_amount_paid;
    }

    /**
     * @return int
     */
    public function length()
    {
        return Frequency::dayDiff($this->payment_date, $this->period_start_date);
    }

    /**
     * @return DateTime
     */
    public function start()
    {
        return $this->period_start_date;
    }

    /**
     * @return DateTime
     */
    public function end()
    {
        return $this->payment_date;
    }

    /**
     * @return float
     */
    public function annual()
    {
        return $this->annual_interest_rate;
    }

    /**
     * @return float
     */
    public function payment()
    {
        return $this->payment_amount;
    }

    /**
     * @param float $pp
     * @param float $ip
     * @param float $tap
     */
    public function lock($pp, $ip, $tap)
    {
        if (!$this->locked) {
            $this->principal_amount_paid      = $pp;
            $this->interest_amount_paid       = $ip;
            $this->total_amount_after_payment = $tap;

            $this->locked = true;
        }
    }

    public function isOnTime(DateTime $dt)
    {
        return ($dt <= $this->payment_date);
    }

    /**
     * @param ScheduledPayment $sp
     * @return int
     */
    public function dayDiff(ScheduledPayment $sp)
    {
        return Frequency::dayDiff($this->payment_date, $sp->payment_date);
    }

    /**
     * @return array
     */
    public function toArray()
    {
        $rt = array(
            'payment_date'           => $this->payment_date->format("Y-m-d"),
            'principal_amount'       => $this->principal_amount_paid,
            'interest_amount'        => $this->interest_amount_paid,
            'total_amount_remaining' => $this->total_amount_after_payment,
            'amount_this_payment'    => $this->principal_amount_paid + $this->interest_amount_paid,
        );

        return $rt;
    }
}