<?php

namespace DPP\Financial\Time;

use \DateTime;
use \DateInterval;

use DPP\Financial\FinancialException;

use DPP\Financial\Time\Holiday\Holiday;

/**
 * Class Business
 *
 * @package DPP\Amortization\Time
 */
class Business extends Frequency
{
    private $holidays;
    private $weekends;

    /**
     * Business constructor.
     */
    public function __construct()
    {
        parent::__construct("Business", "business");

        $this->holidays = Holiday::getList();
        $this->weekends = array(0 => true, 6 => true);
    }

    /**
     * @return float|int
     */
    public function partOfYear()
    {
        return (1 / 365);
    }

    /**
     * @param DateTime $dt
     * @return DateTime
     * @throws FinancialException
     */
    public function getNextDate(DateTime $dt)
    {
        $rt = $this->copy($dt);
        $rt->add(DateInterval::createFromDateString("1 day"));

        while ($this->isWeekend($rt) || $this->isHoliday($rt))
        {
            $rt->add(DateInterval::createFromDateString("1 day"));
        }

        return $rt;
    }

    /**
     * @return float|int
     */
    public function termsInAYear()
    {
        return (365 - (52 * 2) - count($this->holidays));
    }

    /**
     * @return int
     */
    public function daysInTerm()
    {
        return 5;
    }

    /**
     * @param DateTime $dt
     * @return bool
     */
    private function isWeekend(DateTime $dt)
    {
        $dow = $dt->format("w");

        return array_key_exists($dow, $this->weekends);
    }

    /**
     * @param DateTime $dt
     * @return bool
     */
    private function isHoliday(DateTime $dt)
    {
        $rt = false;

        foreach ($this->holidays as $h)
        {
            if ($h->isHoliday($dt))
            {
                $rt = true;
                break;
            }
        }

        return $rt;
    }
}
