<?php

namespace DPP\Financial\Service;

use DPP\Financial\Schedule;
use DPP\Financial\FinancialException;
use DPP\Financial\Time\Frequency;

/**
 * Class ServiceProcessor
 *
 * @package DPP\Financial\Service
 */
class ServiceProcessor
{
    /**
     * @var Schedule
     */
    private $schedule;

    /**
     * @var Payment[]
     */
    private $payments;

    private $paid_items;
    private $total_paid;

    /**
     * ServiceProcessor constructor.
     */
    public function __construct()
    {
        $this->schedule;
        $this->payments = array();

        $this->paid_items = 0;
        $this->total_paid = 0.0;
    }

    /**
     * @param Schedule  $s
     * @param Payment[] $ps
     * @return ServiceProcessor
     */
    public static function create(Schedule $s, array $ps = array())
    {
        $rt = new ServiceProcessor();
        $pmt = $s->payments();

        $rt->schedule = $s;
        $ct = count($pmt);

        for($i=0; $i<$ct; $i++)
        {
            if(!empty($ps[$i]))
            {
                $rt->updateTotals();
            }
        }

        return $rt;
    }

    /**
     * @return Payment[]
     */
    public function payments()
    {
        return $this->payments;
    }

    /**
     * @param float $p
     * @param integer $pn
     * @throws FinancialException
     */
    public function makePayment($amt, $pn = null)
    {
        if($this->paidOff())
        {
            throw new FinancialException("Attempting to make payments on debt that is paid off");
        }

        $pn = ($pn !== null) ? $pn - 1 : $this->paid_items;
        $pmt = $this->schedule->payments();

        if(empty($pmt[$pn]))
        {
            throw new FinancialException("Scheduled payment does not exist");
        }

        $this->payments[] = new Payment($pmt[$pn], Frequency::getCurrentDate(), $amt);
        $this->updateTotals();
    }

    /**
     * @param Payment $p
     * @param integer $pn
     */
    public function reversePayment(Payment $p, $pn)
    {
        if(isset($this->payments[$pn]))
        {
            $this->payments[$pn] = null;
            $this->updateTotals();
        }
    }

    /**
     * @return bool
     */
    public function paidOff()
    {
        return ($this->total_paid >= $this->schedule->principalTotal());
    }

    /**
     * @return array
     */
    public function getStats()
    {
        $rt = array(
            'expected_payments'     => $this->schedule->paymentCount(),
            'number_of_payments'    => count($this->payments),
            'total_paid'            => $this->total_paid,
            'paid_off'              => $this->paidOff()
        );

        return $rt;
    }

    private function updateTotals()
    {
        $tp = 0.0;
        $tc = count($this->payments);

        for($i=0; $i<$tc; $i++)
        {
            $tp += $this->payments[$i]->principalPaid();
        }

        $this->total_paid = $tp;
        $this->paid_items = $tc;
    }
}
