<?php

namespace DPP\Financial\Amortization;

use \DateTime;

use DPP\Financial\Factorization\FactorizationCalculator;
use DPP\Financial\FinancialException;
use DPP\Financial\IncrementType;
use DPP\Financial\PresentValueOfAnnuity;
use DPP\Financial\ScheduledPayment;
use DPP\Financial\Schedule;

use DPP\Financial\Time\Frequency;

/**
 * Class AmortizationCalculator
 *
 * @package DPP\Financial\Amortization
 */
abstract class AmortizationCalculator
{
    const CURRENCY = "$";
    const AD_HOC   = "ad-hoc";
    const STANDARD = "standard";

    /**
     * @var Frequency
     */
    protected $payment_frequency;
    /**
     * @var IncrementType
     */
    protected $increment_type;

    protected $total_amount;
    protected $annual_interest_rate;
    protected $term_count;
    protected $amortization_term_count;
    protected $compound_frequency;
    protected $payment_amount;
    protected $initial_date;
    protected $first_payment_date;
    protected $last_payment_date;
    protected $increment_value;

    private $mode;

    /**
     * @param Method             $mtd Amortization method
     * @param float              $amt Initial total amount
     * @param float              $air Annual interest rate
     * @param Frequency          $pf  Payment frequency
     * @param DateTime|null      $id  Initial date
     * @param DateTime|null      $fpd First payment date
     * @param DateTime|null      $lpd Last payment date
     * @param int|null           $tc  Term count
     * @param float|null         $pa  Payment amount
     * @param Frequency|null     $cp  Compounding frequency
     * @param IncrementType|null $it  Increment type
     * @param float|null         $iv  Increment value
     * @param int|null           $atc Amortization term count
     * @return AmortizationCalculator|null
     * @throws FinancialException
     */
    final public static function create(Method $mtd, $amt, $air, Frequency $pf, DateTime $id = null, DateTime $fpd = null, DateTime $lpd = null, $tc = null, $pa = null, Frequency $cp = null, IncrementType $it = null, $iv = null, $atc = null)
    {
        $nm = $mtd->className();

        if($nm === "factorization")
        {
            $rt = FactorizationCalculator::create($air, $amt, $tc, $pf);
        }
        else
        {
            $rts = sprintf("DPP\Financial\Amortization\%s", $nm);

            $rt = new $rts();

            $rt->total_amount            = $amt;
            $rt->annual_interest_rate    = $air;
            $rt->payment_frequency       = $pf;
            $rt->term_count              = $tc;
            $rt->amortization_term_count = $atc;
            $rt->compound_frequency      = $cp;
            $rt->payment_amount          = $pa;

            if ($id !== null)
            {
                $id->setTime(0, 0, 0);
            }

            if ($fpd !== null)
            {
                $fpd->setTime(0, 0, 0);
            }

            if ($lpd !== null)
            {
                $lpd->setTime(0, 0, 0);
            }

            if (($id !== null) && ($fpd === null))
            {
                $rt->first_payment_date = $rt->initial_date;
            }

            $rt->initial_date       = $id;
            $rt->first_payment_date = $fpd;
            $rt->last_payment_date  = $lpd;

            $rt->increment_type  = $it;
            $rt->increment_value = $iv;

            $nd = ($pa !== null) && ($fpd !== null);
            $fd = ($tc === null) && (($fpd !== null) && ($lpd !== null));
            $ft = ($tc !== null);

            if ((!$fd) && (!$ft) && (!$nd))
            {
                throw new FinancialException("Insufficient information to calculate number of terms");
            }

            if ($fd || $ft)
            {
                $rt->mode = self::STANDARD;
            }
            else
            {
                $rt->mode = self::AD_HOC;
            }

            if (($id !== null) && ($fpd !== null) && ($id > $fpd))
            {
                throw new FinancialException("Initial date cannot come after first payment date");
            }
        }

        return $rt;
    }

    /**
     * @return int|null
     */
    public function getTermCount()
    {
        return $this->term_count;
    }

    /**
     * @return DateTime[]
     * @throws FinancialException
     */
    public function getSchedule()
    {
        $dts = array();

        if ($this->initial_date !== $this->first_payment_date)
        {
            $dts[] = new ScheduledPayment(Frequency::copyDate($this->first_payment_date), Frequency::copyDate($this->initial_date), $this->annual_interest_rate);
        }
        else
        {
            $dts[] = new ScheduledPayment(Frequency::copyDate($this->first_payment_date), Frequency::copyDate($this->first_payment_date), $this->annual_interest_rate);
        }

        $fd  = Frequency::copyDate($this->first_payment_date);
        $pfd = Frequency::copyDate($this->first_payment_date);

        if ($this->payment_amount === null)
        {
            $tat = $this->total_amount;
            $nop = $this->amortization_term_count ?: $this->term_count ?: null;

            $pva                  = new PresentValueOfAnnuity($this->payment_frequency, $this->annual_interest_rate, $nop, $this->compound_frequency);
            $this->payment_amount = $pva->getPeriodicPayment($tat);
        }

        $tc = $this->term_count ?: null;

        if ($tc !== null)
        {
            $ntc = $tc - count($dts);

            while ($ntc > 0)
            {
                $ar = $this->annual_interest_rate;
                $pd = $this->payment_frequency->getNextDate($fd);

                if ($this->increment_type !== null)
                {
                    $ydf = Frequency::yearDiff($pfd, $pd);
                    $ar  += floor($ydf / $this->increment_type->years()) * $this->increment_value;
                }

                $dts[] = new ScheduledPayment($pd, $fd, $ar);

                $fd = $pd;
                $ntc--;
            }
        }
        else
        {
            while ($fd < $this->last_payment_date)
            {
                $ar = $this->annual_interest_rate;
                $pd = $this->payment_frequency->getNextDate($fd);

                if ($this->increment_type !== null)
                {
                    $ydf = Frequency::yearDiff($pfd, $pd);
                    $ar  += floor($ydf / $this->increment_type->years()) * $this->increment_value;
                }

                $rp = $this->payment_frequency->partOfYear() * $ar;

                $dts[] = new ScheduledPayment($pd, $fd, $rp);
                $fd    = $pd;
            }
        }

        if ($this->term_count === null)
        {
            $this->term_count = count($dts);
        }

        return $dts;
    }

    /**
     * @return Schedule
     * @throws FinancialException
     */
    public function run()
    {
        $rt  = new Schedule(self::CURRENCY);
        $ncs = null;

        if ($this->mode === self::AD_HOC)
        {
            $ncs = $this->adHocSchedule();
        }
        else
        {
            $sch = $this->getSchedule();
            $ncs = $this->fixedSchedule($sch);
        }

        $rt->addAll($ncs);
        $rt->lock();

        return $rt;
    }

    /**
     * @param DateTime[] $ds
     * @return ScheduledPayment[]
     */
    abstract protected function fixedSchedule(array $ds);

    /**
     * @return ScheduledPayment[]
     */
    abstract protected function adHocSchedule();
}