<?php

namespace DPP\Financial\Amortization;

use \DateTime;

use DPP\Financial\FinancialException;
use DPP\Financial\PresentValueOfAnnuity;
use DPP\Financial\Interest\Interest;
use DPP\Financial\ScheduledPayment;

use DPP\Financial\Time\Frequency;

/**
 * Class NormalAmortizationCalculator
 *
 * @package DPP\Financial\Amortization
 */
class NormalAmortizationCalculator extends AmortizationCalculator
{
    const NAME = "Normal";
    const SHORT_NAME = "normal";
    const CLASS_NAME = "NormalAmortizationCalculator";

    private $last_total;
    private $increasing_count = 0;

    /**
     * @param DateTime[] $ds
     * @return ScheduledPayment[]
     * @throws FinancialException
     */
    protected function fixedSchedule(array $ds)
    {
        $rt = array();
        $ft = $this->total_amount;

        $np = count($ds);
        $oa = $ds[0]->annual();
        $pa = $this->payment_amount;

        foreach ($ds as $i => $d)
        {
            $da  = $d->annual();
            $ip  = null;
            $dt  = $d->date();
            $tpa = null;

            if (($this->increment_type !== null) && ($oa !== $da))
            {
                $nop = $this->amortization_term_count ?: ($np - $i) ?: null;

                $pva = new PresentValueOfAnnuity($this->payment_frequency, $da, $nop, $this->compound_frequency);
                $pa  = $pva->getPeriodicPayment($ft);

                $oa = $da;
            }

            $pp = null;

            if (($this->initial_date < $dt) && ($i === 0))
            {
                $io = Interest::Get($d->annual(), $ft, $this->payment_frequency->partOfYear(), $this->compound_frequency);
                $ip = $io->overDateRange($this->initial_date, $dt);

                if ($ip > $pa)
                {
                    $tpa = $ip;
                }
            }
            else
            {
                $io = Interest::Get($d->annual(), $ft, $this->payment_frequency->partOfYear(), $this->compound_frequency);
                $ip = $io->value();
            }

            if ($ft >= $pa)
            {
                $pp = round(($tpa ?: $pa) - $ip, Interest::ROUNDING_PRECISION);
            }
            else
            {
                $pp = round($ft, Interest::ROUNDING_PRECISION);
            }

            $ft = round($ft - $pp, Interest::ROUNDING_PRECISION);

            // On last payment, add to principal and zero out the remaining balance
            if (count($rt) === (count($ds) - 1))
            {
                $pp += $ft;
                $ft = 0.0;
            }

            $d->lock($pp, $ip, $ft);
            $rt[] = $d;

            if ($ft <= 0.0)
            {
                break;
            }

            $this->determineRunaway($ft);
        }

        return $rt;
    }

    /**
     * @return ScheduledPayment[]
     * @throws FinancialException
     */
    protected function adHocSchedule()
    {
        $rt = array();

        if ($this->increment_type !== null)
        {
            throw new FinancialException("Unable to calculate ad-hoc ARM schedule");
        }

        $od = Frequency::copyDate($this->initial_date);
        $nd = Frequency::copyDate($this->first_payment_date);
        $ft = $this->total_amount;

        while ($ft > 0.0)
        {

            $io = Interest::Get($this->annual_interest_rate, $ft, $this->payment_frequency->partOfYear(), $this->compound_frequency);
            $ip = $io->value();

            $pp = round($this->payment_amount - $ip, Interest::ROUNDING_PRECISION);

            if ($ft < $this->payment_amount)
            {
                $pp = round($ft, Interest::ROUNDING_PRECISION);
            }

            $ft = round($ft - $pp, Interest::ROUNDING_PRECISION);
            $ns = new ScheduledPayment($nd, $od, $this->annual_interest_rate);

            $ns->lock($pp, $ip, $ft);
            $rt[] = $ns;

            $od = $nd;
            $nd = $this->payment_frequency->getNextDate($nd);

            $this->determineRunaway($ft);
        }

        $this->term_count = count($rt);

        return $rt;
    }

    /**
     * @param float $ft
     * @throws FinancialException
     */
    private function determineRunaway($ft)
    {
        if ($this->last_total !== null)
        {
            if ($ft >= $this->last_total)
            {
                $this->increasing_count++;
            }

            if ($this->increasing_count > 10)
            {
                throw new FinancialException("Runaway Interest, payment too small");
            }
        }

        $this->last_total = $ft;
    }
}