<?php

namespace DPP\Financial\Amortization;

use \DateTime;

use DPP\Financial\FinancialException;
use DPP\Financial\Interest\Interest;
use DPP\Financial\ScheduledPayment;

use DPP\Financial\Time\Frequency;

/**
 * Class FixedPrincipalAmortizationCalculator
 *
 * @package DPP\Financial\Amortization
 */
class FixedPrincipalAmortizationCalculator extends AmortizationCalculator
{
    const NAME = "Fixed Principal";
    const SHORT_NAME = "fixed_principal";
    const CLASS_NAME = "FixedPrincipalAmortizationCalculator";

    /**
     * @param DateTime[] $ds
     * @return ScheduledPayment[]
     * @throws FinancialException
     */
    protected function fixedSchedule(array $ds)
    {
        $rt = array();
        $ft = $this->total_amount;
        $pa = $this->payment_amount;

        foreach ($ds as $d)
        {
            $ip = null;

            $io = Interest::Get($d->annual(), $ft, $this->payment_frequency->partOfYear(), $this->compound_frequency);
            $ip = $io->value();

            $pm = round($pa + $ip, Interest::ROUNDING_PRECISION);

            if ($ft < $pm)
            {
                $ip = 0;
                $pa = $ft;
            }

            $ft -= $pa;

            // On last payment, add to principal and zero out the remaining balance
            if (count($rt) === (count($ds) - 1))
            {
                $pa += $ft;
                $ft = 0.0;
            }

            $d->lock($pa, $ip, $ft);
            $rt[] = $d;

            if ($ft <= 0.0)
            {
                break;
            }
        }

        return $rt;
    }

    /**
     * @return ScheduledPayment[]
     * @throws FinancialException
     */
    protected function adHocSchedule()
    {
        $rt = array();

        $nd = Frequency::copyDate($this->first_payment_date);
        $ft = $this->total_amount;
        $pa = $this->payment_amount;
        $od = null;

        while ($ft > 0.0) {
            $ip = null;

            $io = Interest::Get($this->annual_interest_rate, $ft, $this->payment_frequency->partOfYear(), $this->compound_frequency);
            $ip = $io->value();

            if ($ft < $this->payment_amount) {
                $ip = 0;
                $pa = $ft;
            }

            $ft -= $pa;
            $ns = new ScheduledPayment($nd, $od, $this->annual_interest_rate);

            $ns->lock($pa, $ip, $ft);
            $rt[] = $ns;

            $od = $nd;
            $nd = $this->payment_frequency->getNextDate($nd);
        }

        return $rt;
    }
}