<?php

namespace DPP\Financial\Time;

use \DateTime;
use \DateInterval;

use DPP\Financial\FinancialException;

/**
 * Class Monthly
 *
 * @package DPP\Financial\Time
 */
class Monthly extends Frequency
{
    private $dayOfMonth;

    /**
     * Monthly constructor.
     */
    public function __construct()
    {
        parent::__construct("Monthly", "monthly");

        $this->dayOfMonth = null;
    }

    /**
     * @return float|int
     */
    public function partOfYear()
    {
        return (1 / 12);
    }

    /**
     * @param DateTime $dt
     * @return DateTime
     * @throws FinancialException
     */
    public function getNextDate(DateTime $dt)
    {
        $rt = $this->copy($dt);

        if($this->dayOfMonth == null)
        {
            $this->dayOfMonth = $dt->format("d");
        }

        if($this->isLastDayOfMonth($rt))
        {
            $rt->modify("first day of next month");
            $rt->modify("first day of next month");
            $rt->modify("-1 day");
        }
        else
        {
            $rt->modify("first day of next month");
            $dim = cal_days_in_month(CAL_GREGORIAN, (int) $rt->format("m"), (int) $rt->format("Y"));

            if($this->dayOfMonth > $dim)
            {
                $sdt = $dim - 1;
                $rt->modify("+$sdt days");
            }
            else
            {
                $sdt = $this->dayOfMonth - 1;
                $rt->modify("+$sdt days");
            }
        }

        return $rt;
    }

    /**
     * @return int
     */
    public function termsInAYear()
    {
        return 12;
    }

    /**
     * @return int
     */
    public function daysInTerm()
    {
        return 30;
    }

    private function isLastDayOfMonth(DateTime $dt)
    {
        $dta = clone $dt;
        $m = $dta->format("m");
        $dta->modify("+1 day");
        $ma = $dta->format("m");

        return ($m != $ma);
    }
}
