<?php

namespace DPP\Financial\Time;

use \DateTime;
use \DateInterval;
use \JsonSerializable;

use DPP\Financial\FinancialException;

abstract class Frequency implements JsonSerializable
{
    const DAYS_365 = 365;
    const DAYS_360 = 360;
    const DAYS_364 = 364;

    private $name;
    private $shortName;

    protected $lastDateInterval;
    protected $daysInYear;

    public function __construct($n, $sn, $da = self::DAYS_365)
    {
        $this->name      = $n;
        $this->shortName = $sn;

        $this->daysInYear = $da;
    }

    final public static function getList()
    {
        $rt = array();

        $rt[] = new Continuous();
        $rt[] = new Daily();
        $rt[] = new Business();
        $rt[] = new Weekly();
        $rt[] = new BiWeekly();
        $rt[] = new SemiMonthly();
        $rt[] = new Monthly();
        $rt[] = new Quarterly();
        $rt[] = new SemiAnnually();
        $rt[] = new Annually();
        $rt[] = new ExactDays();

        return $rt;
    }

    final public static function create($n, DateTime $fd = null, DateTime $ld = null, $da = null, $dr = null)
    {
        $rt  = null;
        $rst = self::getList();

        foreach ($rst as $r)
        {
            if ($r->shortName() === $n)
            {
                $rt = $r;

                if($rt instanceof SemiMonthly)
                {
                    $rt = new SemiMonthly($fd, $ld);
                }

                if($rt instanceof ExactDays)
                {
                    $rt = new ExactDays($da);
                }

                if($dr !== null)
                {
                    $rt->setDaysInYear($dr);
                }

                break;
            }
        }

        if ($rt === null)
        {
            throw new FinancialException(sprintf("Unable to create type '%s'", $n));
        }

        return $rt;
    }

    final public static function getCurrentDate()
    {
        $rt = new DateTime();

        $rt->setTime(0, 0, 0);

        return $rt;
    }

    final public static function dayDiff(DateTime $fdt, DateTime $sdt)
    {
        return (int) $fdt->diff($sdt)->format("%a");
    }

    final public static function yearDiff(DateTime $fdt, DateTime $sdt)
    {
        return (int) $fdt->diff($sdt)->format("%y");
    }

    final public static function getFrequency($fs)
    {
        $rt  = null;
        $frq = self::getList();

        foreach ($frq as $f)
        {
            if ($f->shortName() === $fs)
            {
                $rt = $f;
                break;
            }
        }

        if ($rt === null)
        {
            throw new FinancialException(sprintf("Frequency string '%s' is invalid", $fs));
        }

        return $rt;
    }

    final public static function generateSchedule(DateTime $sd, Frequency $f, $nt = null, DateTime $sod = null)
    {
        $rt = array();
        $nd = clone $sd;
        $nt = (int) $nt;

        if(($nt === null) && ($sod === null))
        {
            throw new FinancialException("Need the number of terms or a stop date ");
        }

        if(!empty($nt))
        {
            while ($nt > 0)
            {
                $rt[] = $nd;
                $nd   = $f->getNextDate($nd);

                $nt--;
            }
        }
        else if(!empty($sod))
        {
            while ($nd <= $sod)
            {
                $rt[] = $nd;
                $nd   = $f->getNextDate($nd);
            }
        }

        return $rt;
    }

    public function name()
    {
        return $this->name;
    }

    public function setDaysInYear($da)
    {
        if(in_array($da, array(self::DAYS_365, self::DAYS_360, self::DAYS_364)))
        {
            $this->daysInYear = $da;
        }
    }

    public function shortName()
    {
        return $this->shortName;
    }

    public function diffWith(Frequency $f)
    {
        $rt = null;

        $tpo = $this->partOfYear();
        $fpo = $f->partOfYear();

        if ($fpo > 0.0)
        {
            $rt = $tpo / $fpo;
        }

        return $rt;
    }

    protected function copy(DateTime $dt)
    {
        return clone $dt;
    }

    public function jsonSerialize()
    {
        return $this->shortName();
    }

    public function daysInYear()
    {
        return $this->daysInYear;
    }

    abstract public function partOfYear();

    abstract public function getNextDate(DateTime $dt);

    abstract public function termsInAYear();

    abstract public function daysInTerm();
}
