<?php

namespace DPP\Financial\Scheduling\Amortization;

use \DateTime;

use DPP\Financial\FinancialException;
use DPP\Financial\Interest\Interest;
use DPP\Financial\Scheduling\ScheduledPayment;
use DPP\Financial\Scheduling\Scheduler;

use DPP\Financial\Time\Frequency;

class NormalAmortizationScheduler extends Scheduler
{
    const DEFAULT_TOLERANCE = 0.02;
    const NAME = "Normal";
    const SHORT_NAME = "normal";
    const CLASS_NAME = "NormalAmortizationScheduler";

    private $lastTotal;
    private $increasingCount = 0;

    public function __construct($amt, $air, Frequency $pf, DateTime $id, DateTime $fd, $pa = null, Frequency $cp = null)
    {
        parent::__construct($amt, $air, $pf, $id, $fd, $pa, $cp);

        $this->lastTotal = 0.0;
        $this->increasingCount = 0;
        $this->tolerance = self::DEFAULT_TOLERANCE;
    }

    protected function fixedScheduleFinal(array $dts)
    {
        $rt = null;

        $cta = count($dts);
        $fp = $this->paymentAmount;
        $pa = $this->paymentAmount;

        $ap = 0.0;
        $rpt = false;

        $lmt = 19;
        $iat = 0;

        do
        {
            $opa = $pa;
            $pa = $this->round(($rpt) ? ((($cta - 1) * $pa) + $ap) / $cta : $pa);

            // no point in doing it again with the same one as before
            // or, if they're so close, then average and continue;
            if(($rpt) && ($opa == $pa))
            {
                break;
            }
            else if(($rpt) && ($opa - $pa) < $this->tolerance)
            {
                $pa = ($opa + $pa) / 2;
            }


            $rt = array();
            $ft = $this->totalAmount;
            $pd = null;

            foreach($dts as $i => $d)
            {
                $da  = $d->annual();
                $dt  = $d->date();
                $nd = ($pd != null) ? $dt->diff($pd)->days : $dt->diff($this->initialDate)->days;

                $tpa = null;
                $ip  = null;
                $pp = null;
                $ngoia = $nd / $this->paymentFrequency->daysInYear();

                $io = Interest::Get($da, $ft, $ngoia, $this->compoundFrequency);

                if(($this->initialDate < $dt) && ($i === 0))
                {
                    $ip = $io->overDateRange($this->initialDate, $dt);

                    if ($ip > $pa)
                    {
                        $pp = ($pa - $ip);
                    }
                }
                else
                {
                    $ip = $io->value();
                }

                if($ft >= $pa)
                {
                    $pp = $pp ?: ($pa - $ip);
                }
                else
                {
                    $pp = $ft;
                }

                $pp = $this->round($pp);
                $ip = $this->round($ip);
                $ft = $this->round($ft - $pp);


                // On last payment, add to principal and zero out the remaining balance
                if (count($rt) === (count($dts) - 1))
                {
                    $pp += $ft;
                    $ft = 0.0;
                }

                $ap = ($pp + $ip);
                $d->setFinal($pp, $ip, $ft);
                $rt[] = $d;

                if ($ft <= 0.0)
                {
                    break;
                }

                $this->determineRunaway($ft);

                $pd = $dt;
            }

            $rpt = true;

            $this->increasingCount = 0;
            $this->lastTotal = null;

            if($iat++ > $lmt)
            {
                break;
            }

        } while((abs($fp - $ap) > $this->tolerance) && ($this->figured));

        foreach($dts as $i => $d)
        {
            if($i === ($cta - 1))
            {
                $pp = $d->principal();
                $ip = $d->interest();
                $ft = $d->balance();
                $am = $d->amount();

                //$d->setFinal($pp, $ip - ($am - $pa), $ft); // adjust final interest by difference between the normal payment and the final payment
            }

            $d->lock();
        }

        return $rt;
    }

    protected function adHocScheduleFinal()
    {
        $rt = array();

        $od = Frequency::copyDate($this->initial_date);
        $nd = Frequency::copyDate($this->first_payment_date);
        $ft = $this->total_amount;

        while ($ft > 0.0)
        {

            $io = Interest::Get($this->annual_interest_rate, $ft, $this->payment_frequency->partOfYear(), $this->compound_frequency);
            $ip = $io->value();

            $pp = $this->payment_amount - $ip;

            if ($ft < $this->payment_amount)
            {
                $pp = $ft;
            }

            $ip = $this->round($ip);
            $pp = $this->round($pp);
            $ft = $this->round($ft - $pp);

            $ns = new ScheduledPayment($nd, $od, $this->annual_interest_rate);

            $ns->lock($pp, $ip, $ft);
            $rt[] = $ns;

            $od = $nd;
            $nd = $this->payment_frequency->getNextDate($nd);

            $this->determineRunaway($ft);
        }

        $this->term_count = count($rt);

        return $rt;
    }

    private function determineRunaway($ft)
    {
        if ($this->lastTotal !== null)
        {
            if ($ft >= $this->lastTotal)
            {
                $this->increasingCount++;
            }

            if ($this->increasingCount > 10)
            {
                throw new FinancialException("Runaway Interest, payment too small");
            }
        }

        $this->lastTotal = $ft;
    }
}