<?php

namespace DPP\Financial\Scheduling;

use \DateTime;

use DPP\Financial\FinancialException;
use DPP\Financial\PresentValueOfAnnuity;

use DPP\Financial\Time\Frequency;

abstract class Scheduler
{
    const CURRENCY = "$";
    const AD_HOC = "AD_HOC";
    const NORMAL = "NORMAL";

    protected $paymentFrequency;
    protected $totalAmount;
    protected $annualInterestRate;
    protected $compoundFrequency;
    protected $paymentAmount;
    protected $initialDate;
    protected $firstPaymentDate;

    protected function __construct($amt, $air, Frequency $pf, DateTime $id, DateTime $fd = null, $pa = null, Frequency $cp = null)
    {
        $this->totalAmount = $amt;
        $this->annualInterestRate = $air;
        $this->paymentFrequency = $pf;
        $this->compoundFrequency = $cp;
        $this->paymentAmount = $pa;

        $this->initialDate = $id;
        $this->firstPaymentDate = $fd ?: $id;
    }

    protected function getDateSeriesByTermCount($tc)
    {
        $dts = array();
        $dts[] = new ScheduledPayment(clone $this->firstPaymentDate, clone $this->initialDate, $this->annualInterestRate);

        $fd  = clone $this->firstPaymentDate;

        if($this->paymentAmount === null)
        {
            $pva = new PresentValueOfAnnuity($this->paymentFrequency, $this->annualInterestRate, $tc, $this->compoundFrequency);
            $this->paymentAmount = $pva->getPeriodicPayment($this->totalAmount);
        }

        $ntc = $tc - count($dts);

        while ($ntc > 0)
        {
            $pd = $this->paymentFrequency->getNextDate($fd);
            $dts[] = new ScheduledPayment($pd, $fd, $this->annualInterestRate);

            $fd = $pd;
            $ntc--;
        }

        return $dts;
    }

    protected function getDateSeriesByEndDate(DateTime $dt)
    {
        $fd = clone $this->firstPaymentDate;
        $tc = 1;

        while ($fd < $dt)
        {
            $pd = $this->paymentFrequency->getNextDate($fd);

            if($pd > $dt)
            {
                $pd = clone $dt;
            }

            $tc++;
            $fd = $pd;
        }

        return $this->getDateSeriesByTermCount($tc);
    }

    public function generate($md, $tc = null, DateTime $dt = null)
    {
        $rt  = new Schedule(self::CURRENCY);
        $ncs = null;

        if($md === self::AD_HOC)
        {
            $ncs = $this->adHocScheduleFinal();
        }
        else
        {
            $dts = null;

            if(($tc !== null) && (is_int($tc)) && ($tc > 0))
            {
                $dts = $this->getDateSeriesByTermCount($tc);
            }
            else if($dt !== null)
            {
                $dts = $this->getDateSeriesByEndDate($dt);
            }
            else
            {
                throw new FinancialException("A term count or date is required for schedule generation");
            }

            $ncs = $this->fixedScheduleFinal($dts);
        }

        $rt->addAll($ncs);
        $rt->lock();

        return $rt;
    }

    protected abstract function fixedScheduleFinal(array $dts);
    protected abstract function adHocScheduleFinal();
}