<?php

namespace DPP\Financial\Scheduling;

use DPP\Financial\FinancialException;

class Schedule
{
    private $items;

    private $payment_total;
    private $interest_total;
    private $principal_total;

    private $currency;
    private $locked;

    public function __construct($cu, $ok = false)
    {
        $this->items = array();

        $this->payment_total   = 0.0;
        $this->interest_total  = 0.0;
        $this->principal_total = 0.0;

        $this->currency = $cu;
        $this->locked   = $ok;
    }

    public static function SortByDate(ScheduledPayment $o1, ScheduledPayment $o2)
    {
        $rt = 0;

        if($o1->date() < $o2->date())
        {
            $rt = -1;
        }
        else if($o1->date() > $o2->date())
        {
            $rt = 1;
        }

        return $rt;
    }

    public function sort()
    {
        usort($this->items, array("Schedule", "SortByDate"));
    }

    public function addOn(Schedule $s)
    {
        if($this->locked)
        {
            throw new FinancialException("Unable to modify locked schedule");
        }

        foreach($s->items as $i)
        {
            $this->items[] = $i;
        }
    }

    public function addAll(array $sps)
    {
        if ($this->locked)
        {
            throw new FinancialException("Unable to modify locked Schedule");
        }

        foreach ($sps as $s)
        {
            if ($s instanceof ScheduledPayment)
            {
                $this->add($s);
            }
        }
    }

    public function add(ScheduledPayment $sp, $ow = false)
    {
        if ($this->locked)
        {
            throw new FinancialException("Unable to modify locked Schedule");
        }

        if($ow)
        {
            $tc = count($this->items);
            $fnd = false;

            for($i=0; $i<$tc; $i++)
            {
                if($this->items[$i]->date() == $sp->date())
                {
                    $it = $this->items[$i];

                    $rt = new ScheduledPayment($sp->start(), $sp->date(), $sp->annual(), $sp->payment() + $it->payment());
                    $rt->lock($sp->principal() + $it->principal(), $sp->interest() + $it->interest(), $it->balance() - $sp->principal());

                    $this->items[$i] = $rt;
                    $fnd = true;
                    break;
                }
            }

            if(!$fnd)
            {
                $this->items[] = $sp;
            }
        }
        else
        {
            $this->items[] = $sp;
        }
    }

    public function lock()
    {
        $this->calculateStats();
        $this->locked = true;
    }

    public function isLocked()
    {
        return $this->locked;
    }

    public function paymentTotal()
    {
        return $this->payment_total;
    }

    public function interestTotal()
    {
        return $this->interest_total;
    }

    public function principalTotal()
    {
        return $this->principal_total;
    }

    public function paymentCount()
    {
        return count($this->items);
    }

    public function currency()
    {
        return $this->currency;
    }

    public function payments()
    {
        $this->lock();
        return $this->items;
    }

    public function items()
    {
        return $this->items;
    }

    public function amount()
    {
        $rt = 0.0;

        if (count($this->items) > 0)
        {
            $rt = $this->items[1]->amount();
        }

        return $rt;
    }

    private function calculateStats()
    {
        foreach ($this->items as $i)
        {
            $this->payment_total   += $i->amount();
            $this->interest_total  += $i->interest();
            $this->principal_total += $i->principal();
        }
    }
}