<?php

namespace DPP\Financial\Scheduling\Amortization;

use \DateTime;

use DPP\Financial\FinancialException;
use DPP\Financial\Interest\Interest;
use DPP\Financial\Scheduling\ScheduledPayment;
use DPP\Financial\Scheduling\Scheduler;

use DPP\Financial\Time\Frequency;

class NormalAmortizationScheduler extends Scheduler
{
    const NAME = "Normal";
    const SHORT_NAME = "normal";
    const CLASS_NAME = "NormalAmortizationScheduler";

    private $lastTotal;
    private $increasingCount = 0;

    public function __construct($amt, $air, Frequency $pf, DateTime $id, DateTime $fd, $pa = null, Frequency $cp = null)
    {
        parent::__construct($amt, $air, $pf, $id, $fd, $pa, $cp);

        $this->lastTotal = 0.0;
        $this->increasingCount = 0;
    }

    protected function fixedScheduleFinal(array $dts)
    {
        $rt = array();
        $ft = $this->totalAmount;

        $np = count($dts);
        $oa = $dts[0]->annual();
        $pa = $this->paymentAmount;

        foreach($dts as $i => $d)
        {
            $da  = $d->annual();
            $dt  = $d->date();

            $tpa = null;
            $ip  = null;
            $pp = null;

            $io = Interest::Get($da, $ft, $this->paymentFrequency->partOfYear(), $this->compoundFrequency);

            if(($this->initialDate < $dt) && ($i === 0))
            {
                $ip = $io->overDateRange($this->initialDate, $dt);

                if ($ip > $pa)
                {
                    $tpa = $ip;
                }
            }
            else
            {
                $ip = $io->value();
            }

            if($ft >= $pa)
            {
                $pp = round(($tpa ?: $pa) - $ip, Interest::ROUNDING_PRECISION);
            }
            else
            {
                $pp = round($ft, Interest::ROUNDING_PRECISION);
            }

            $ft = round($ft - $pp, Interest::ROUNDING_PRECISION);

            // On last payment, add to principal and zero out the remaining balance
            if (count($rt) === (count($dts) - 1))
            {
                $pp += $ft;
                $ft = 0.0;
            }

            $d->lock($pp, $ip, $ft);
            $rt[] = $d;

            if ($ft <= 0.0)
            {
                break;
            }

            $this->determineRunaway($ft);
        }

        return $rt;
    }

    protected function adHocScheduleFinal()
    {
        $rt = array();

        $od = Frequency::copyDate($this->initial_date);
        $nd = Frequency::copyDate($this->first_payment_date);
        $ft = $this->total_amount;

        while ($ft > 0.0)
        {

            $io = Interest::Get($this->annual_interest_rate, $ft, $this->payment_frequency->partOfYear(), $this->compound_frequency);
            $ip = $io->value();

            $pp = round($this->payment_amount - $ip, Interest::ROUNDING_PRECISION);

            if ($ft < $this->payment_amount)
            {
                $pp = round($ft, Interest::ROUNDING_PRECISION);
            }

            $ft = round($ft - $pp, Interest::ROUNDING_PRECISION);
            $ns = new ScheduledPayment($nd, $od, $this->annual_interest_rate);

            $ns->lock($pp, $ip, $ft);
            $rt[] = $ns;

            $od = $nd;
            $nd = $this->payment_frequency->getNextDate($nd);

            $this->determineRunaway($ft);
        }

        $this->term_count = count($rt);

        return $rt;
    }

    private function determineRunaway($ft)
    {
        if ($this->lastTotal !== null)
        {
            if ($ft >= $this->lastTotal)
            {
                $this->increasingCount++;
            }

            if ($this->increasingCount > 10)
            {
                throw new FinancialException("Runaway Interest, payment too small");
            }
        }

        $this->lastTotal = $ft;
    }
}