<?php

namespace DPP\Financial\Time;

use \DateTime;
use \DateInterval;

use DPP\Financial\FinancialException;

/**
 * Class Frequency
 *
 * @package DPP\Financial\Time
 */
abstract class Frequency
{
    private $name;
    private $short_name;

    /**
     * @var DateInterval
     */
    protected $lastDateInterval;

    /**
     * Frequency constructor.
     *
     * @param string $n  Name
     * @param string $sn Short name
     */
    public function __construct($n, $sn)
    {
        $this->name       = $n;
        $this->short_name = $sn;
    }

    /**
     * @return self[]
     */
    final public static function getList()
    {
        $rt = array();

        $rt[] = new Continuous();
        $rt[] = new Daily();
        $rt[] = new Business();
        $rt[] = new Weekly();
        $rt[] = new BiWeekly();
        $rt[] = new SemiMonthly();
        $rt[] = new Monthly();
        $rt[] = new Quarterly();
        $rt[] = new SemiAnnually();
        $rt[] = new Annually();

        return $rt;
    }

    /**
     * @param string $n The name of the freqency to create
     * @return Frequency
     * @throws FinancialException
     */
    final public static function create($n, DateTime $fd = null, DateTime $ld = null)
    {
        $rt  = null;
        $rst = self::getList();

        foreach ($rst as $r)
        {
            if ($r->shortName() === $n)
            {
                $rt = $r;

                if($rt instanceof SemiMonthly)
                {
                    $rt = new SemiMonthly($fd, $ld);
                }

                break;
            }
        }

        if ($rt === null)
        {
            throw new FinancialException(sprintf("Unable to create type '%s'", $n));
        }

        return $rt;
    }

    /**
     * @return DateTime
     */
    final public static function getCurrentDate()
    {
        $rt = new DateTime();

        $rt->setTime(0, 0, 0);

        return $rt;
    }

    /**
     * @param DateTime $fdt
     * @param DateTime $sdt
     * @return int
     */
    final public static function dayDiff(DateTime $fdt, DateTime $sdt)
    {
        return (int) $fdt->diff($sdt)->format("%a");
    }

    /**
     * @param DateTime $fdt
     * @param DateTime $sdt
     * @return int
     */
    final public static function yearDiff(DateTime $fdt, DateTime $sdt)
    {
        return (int) $fdt->diff($sdt)->format("%y");
    }

    /**
     * @param string $fs
     * @return Frequency
     * @throws FinancialException
     */
    final public static function getFrequency($fs)
    {
        $rt  = null;
        $frq = self::getList();

        foreach ($frq as $f)
        {
            if ($f->shortName() === $fs)
            {
                $rt = $f;
                break;
            }
        }

        if ($rt === null)
        {
            throw new FinancialException(sprintf("Frequency string '%s' is invalid", $fs));
        }

        return $rt;
    }

    /**
     * @param DateTime      $sd
     * @param Frequency     $f
     * @param integer       $nt
     * @return DateTime[]
     * @throws FinancialException
     */
    final public static function generateSchedule(DateTime $sd, Frequency $f, $nt = null, DateTime $sod = null)
    {
        $rt = array();
        $nd = clone $sd;
        $nt = (int) $nt;

        if(($nt === null) && ($sod === null))
        {
            throw new FinancialException("Need the number of terms or a stop date ");
        }

        if(!empty($nt))
        {
            while ($nt > 0)
            {
                $rt[] = $nd;
                $nd   = $f->getNextDate($nd);

                $nt--;
            }
        }
        else if(!empty($sod))
        {
            while ($nd <= $sod)
            {
                $rt[] = $nd;
                $nd   = $f->getNextDate($nd);
            }
        }

        return $rt;
    }

    /**
     * @return string
     */
    public function name()
    {
        return $this->name;
    }

    /**
     * @return string
     */
    public function shortName()
    {
        return $this->short_name;
    }

    /**
     * @param Frequency $f
     * @return float|int|null
     */
    public function diffWith(Frequency $f)
    {
        $rt = null;

        $tpo = $this->partOfYear();
        $fpo = $f->partOfYear();

        if ($fpo > 0.0)
        {
            $rt = $tpo / $fpo;
        }

        return $rt;
    }

    /**
     * @param DateTime $dt
     * @return DateTime
     * @throws FinancialException
     */
    protected function copy(DateTime $dt)
    {
        return clone $dt;
    }

    /**
     * @return float
     */
    abstract public function partOfYear();

    /**
     * @param DateTime $dt
     * @return DateTime
     */
    abstract public function getNextDate(DateTime $dt);

    /**
     * @return int
     */
    abstract public function termsInAYear();

    /**
     * @return int
     */
    abstract public function daysInTerm();
}
