<?php

namespace DPP\Financial\Scheduling\Amortization;

use \DateTime;

use DPP\Financial\FinancialException;
use DPP\Financial\Interest\Interest;
use DPP\Financial\Scheduling\ScheduledPayment;
use DPP\Financial\Scheduling\Scheduler;

use DPP\Financial\Time\Frequency;

class NormalAmortizationScheduler extends Scheduler
{
    const NAME = "Normal";
    const SHORT_NAME = "normal";
    const CLASS_NAME = "NormalAmortizationScheduler";

    private $lastTotal;
    private $increasingCount = 0;

    public function __construct($amt, $air, Frequency $pf, DateTime $id, DateTime $fd, $pa = null, Frequency $cp = null)
    {
        parent::__construct($amt, $air, $pf, $id, $fd, $pa, $cp);

        $this->lastTotal = 0.0;
        $this->increasingCount = 0;
    }

    protected function fixedScheduleFinal(array $dts)
    {
        $rt = array();
        $ft = $this->totalAmount;

        $np = count($dts);
        $oa = $dts[0]->annual();
        $pa = $this->paymentAmount;

        foreach($dts as $i => $d)
        {
            $da  = $d->annual();
            $dt  = $d->date();

            $tpa = null;
            $ip  = null;
            $pp = null;

            $io = Interest::Get($da, $ft, $this->paymentFrequency->partOfYear(), $this->compoundFrequency);

            if(($this->initialDate < $dt) && ($i === 0))
            {
                $ip = $io->overDateRange($this->initialDate, $dt);

                if ($ip > $pa)
                {
                    $tpa = $ip;
                }
            }
            else
            {
                $ip = $io->value();
            }

            if($ft >= $pa)
            {
                $pp = $this->round($tpa ?: $pa);
            }
            else
            {
                $pp = $this->round($ft);
            }

            $ft = $this->round($ft - $pp);

            // On last payment, add to principal and zero out the remaining balance
            if (count($rt) === (count($dts) - 1))
            {
                $pp += $ft;
                $ft = 0.0;
            }

            $d->lock($pp, $ip, $ft);
            $rt[] = $d;

            if ($ft <= 0.0)
            {
                break;
            }

            $this->determineRunaway($ft);
        }

        return $rt;
    }

    protected function adHocScheduleFinal()
    {
        $rt = array();

        $od = Frequency::copyDate($this->initialDate);
        $nd = Frequency::copyDate($this->firstPaymentDate);
        $ft = $this->totalAmount;

        while ($ft > 0.0)
        {

            $io = Interest::Get($this->annualInterestRate, $ft, $this->paymentFrequency->partOfYear(), $this->compoundFrequency);
            $ip = $io->value();

            $pp = round($this->paymentAmount - $ip, Interest::ROUNDING_PRECISION);

            if ($ft < $this->paymentAmount)
            {
                $pp = round($ft, Interest::ROUNDING_PRECISION);
            }

            $ft = round($ft - $pp, Interest::ROUNDING_PRECISION);
            $ns = new ScheduledPayment($nd, $od, $this->annualInterestRate);

            $ns->lock($pp, $ip, $ft);
            $rt[] = $ns;

            $od = $nd;
            $nd = $this->paymentFrequency->getNextDate($nd);

            $this->determineRunaway($ft);
        }

        $this->term_count = count($rt);

        return $rt;
    }

    private function determineRunaway($ft)
    {
        if ($this->lastTotal !== null)
        {
            if ($ft >= $this->lastTotal)
            {
                $this->increasingCount++;
            }

            if ($this->increasingCount > 10)
            {
                throw new FinancialException("Runaway Interest, payment too small");
            }
        }

        $this->lastTotal = $ft;
    }
}