<?php

namespace DPP\Financial;

use DPP\Financial\Time\Frequency;
use DPP\Financial\Time\Continuous;
use DPP\Financial\Interest\Interest;

/**
 * Class PresentValueOfAnnuity
 *
 * @package DPP\Financial
 */
class PresentValueOfAnnuity
{
    private $frequency;
    private $annual_interest_rate;
    private $number_of_periods;
    private $compound_frequency;

    /**
     * PresentValueOfAnnuity constructor.
     *
     * @param Frequency      $f
     * @param float          $air
     * @param int            $nop
     * @param Frequency|null $cpf
     */
    public function __construct(Frequency $f, $air, $nop, Frequency $cpf = null)
    {
        $this->frequency            = $f;
        $this->annual_interest_rate = $air;
        $this->number_of_periods    = $nop;
        $this->compound_frequency   = $cpf;
    }

    /**
     * @param float $pp
     * @return float
     */
    public function getTotalPayment($pp)
    {
        $rpp = $this->getPeriodicRate();
        $rt  = $pp * ((1 - ((1 + $rpp) ** (-1 * $this->number_of_periods))) / $rpp);

        return round($rt, Interest::ROUNDING_PRECISION);
    }

    /**
     * @param float $toa
     * @return float
     */
    public function getPeriodicPayment($toa)
    {
        $rpp = $this->getPeriodicRate();
        $npp = ((1 + $rpp) ** (-1 * $this->number_of_periods));
        $rt  = ($toa * $rpp) / (1 - $npp);

        return round($rt, Interest::ROUNDING_PRECISION);
    }

    /**
     * @return float
     */
    private function getPeriodicRate()
    {
        return $this->getEAR() * (1 / $this->frequency->termsInAYear());
    }

    /**
     * @return float
     */
    private function getEAR()
    {
        $rt = $this->annual_interest_rate;

        if ($this->compound_frequency !== null)
        {
            if ($this->compound_frequency instanceof Continuous)
            {
                $rt = (M_E ** $this->annual_interest_rate) - 1;
            }
            else
            {
                $ty = $this->compound_frequency->termsInAYear();
                $rt = ((1 + ($this->annual_interest_rate / $ty)) ** $ty) - 1;
            }
        }

        return $rt;
    }
}