<?php

namespace DPP\Financial\Interest;

use \DateTime;

use DPP\Financial\FinancialException;
use DPP\Financial\Time\Frequency;

/**
 * Class BaseInterest
 *
 * @package DPP\Financial\Interest
 */
abstract class Interest
{
    const DEFAULT_DAYS_IN_YEAR = 365;
    const DEFAULT_DAYS_IN_MONTH = 30;

    const ROUNDING_PRECISION = 2;

    /**
     * @var float
     */
    protected $annual_interest_rate;

    /**
     * @var integer
     */
    protected $days_per_year;

    /**
     * @var integer
     */
    protected $days_per_month;

    /**
     * Interest constructor.
     *
     * @param          $air
     * @param integer  $dpy
     * @param integer  $dpm
     */
    public function __construct($air, $dpy = self::DEFAULT_DAYS_IN_YEAR, $dpm = self::DEFAULT_DAYS_IN_MONTH)
    {
        $this->annual_interest_rate  = $air;
        $this->days_per_year = $dpy;
        $this->days_per_month = $dpm;
    }

    /**
     * @param float          $air
     * @param float          $p
     * @param float          $noy
     * @param Frequency|null $f
     * @return CompoundInterest|SimpleInterest
     * @throws FinancialException
     */
    public static function Get($air, $p, $noy, Frequency $f = null)
    {
        $rt = null;

        if($f !== null)
        {
            $rt = new CompoundInterest($air, $p, $noy, $f);
        }
        else
        {
            $rt = new SimpleInterest($air, $p, $noy);
        }

        if($rt === null)
        {
            throw new FinancialException("Unable to get interest");
        }

        return $rt;
    }

    /**
     *
     * @param float     $p
     * @param Frequency $pf
     * @return float
     */
    public function overSingleTerm($p, Frequency $pf)
    {
        $noy = 1.0 / $pf->termsInAYear();

        return $this->calculate($noy);
    }

    /**
     * @param float    $p
     * @param DateTime $sd
     * @param DateTime $fd
     * @return float
     */
    public function overDateRange(DateTime $sd, DateTime $fd)
    {
        $noy = Frequency::dayDiff($sd, $fd) / $this->days_per_year;

        return $this->calculate($noy);
    }

    /**
     * @return float
     */
    public function value()
    {
        return round($this->calculate(), self::ROUNDING_PRECISION);
    }

    /**
     * @param Frequency $f
     * @return float
     */
    protected function getInterestPartOfFrequency(Frequency $f)
    {
        return ($this->annual_interest_rate * ($f->daysInTerm() / $this->days_per_year));
    }

    /**
     * @return float
     */
    abstract protected function calculate($noy = null);
}