<?php

namespace DPP\Financial;

use DPP\Financial\FinancialException as FinancialException;

/**
 * Class Schedule
 *
 * @package DPP\Financial
 */
class Schedule
{
    private $items;

    private $paymentTotal;
    private $interestTotal;
    private $principalTotal;

    private $currency;
    private $locked;

    /**
     * Schedule constructor.
     *
     * @param string $cu
     */
    public function __construct($cu)
    {
        $this->items = array();

        $this->paymentTotal = 0.0;
        $this->interestTotal = 0.0;
        $this->principalTotal = 0.0;

        $this->currency = $cu;
        $this->locked = false;
    }

    /**
     * @param ScheduledPayment[] $sps
     * @throws FinancialException
     */
    public function addAll(array $sps)
    {
        if($this->locked)
        {
            throw new FinancialException("Unable to modify locked Schedule");
        }

        foreach($sps as $s)
        {
            if($s instanceof ScheduledPayment)
            {
                $this->items[] = $s;
            }
        }
    }

    /**
     * @param ScheduledPayment $sp
     * @throws FinancialException
     */
    public function add(ScheduledPayment $sp)
    {
        if($this->locked)
        {
            throw new FinancialException("Unable to modify locked Schedule");
        }

        $this->items[] = $sp;
    }

    /**
     *
     */
    public function lock()
    {
        $this->calculateStats();
        $this->locked = true;
    }

    /**
     * @return float
     */
    public function paymentTotal()
    {
        return $this->paymentTotal;
    }

    /**
     * @return float
     */
    public function interestTotal()
    {
        return $this->interestTotal;
    }

    /**
     * @return float
     */
    public function principalTotal()
    {
        return $this->principalTotal;
    }

    /**
     * @return int
     */
    public function paymentCount()
    {
        return count($this->items);
    }

    /**
     * @return string
     */
    public function currency()
    {
        return $this->currency;
    }

    /**
     * @return ScheduledPayment[]
     */
    public function payments()
    {
        $this->lock();
        return $this->items;
    }

    /**
     * @return float
     */
    public function amount()
    {
        $rt = 0.0;

        if(count($this->items) > 0)
        {
            $rt = $this->items[1]->amount();
        }

        return $rt;
    }

    /**
     *
     */
    private function calculateStats()
    {
        foreach($this->items as $i)
        {
            $this->paymentTotal += $i->amount();
            $this->interestTotal += $i->interest();
            $this->principalTotal += $i->principal();
        }
    }
}