<?php

namespace DPP\Financial;

use \DateTime as DateTime;

use DPP\Financial\Time\Continuous as Continuous;
use DPP\Financial\Time\Frequency as Frequency;

/**
 * Class Interest
 *
 * @package DPP\Financial
 */
class Interest
{
    const ROUNDING_PRECISION = 2;

    private $annualInterestRate;
    private $compoundingFrequency;

    /**
     * Interest constructor.
     *
     * @param float          $air
     * @param Frequency|null $f
     */
    public function __construct($air, Frequency $f = null)
    {
        $this->annualInterestRate = $air;
        $this->compoundingFrequency = $f;
    }

    /**
     * @param float     $p Principal
     * @param float     $noy Number of years
     * @return float
     */
    public function simple($p, $noy)
    {
        $ip = round($p * ($this->annualInterestRate * $noy), self::ROUNDING_PRECISION);

        return $ip;
    }

    /**
     * @param float     $p Principal
     * @param Frequency $pf Term frequency
     * @return float|null
     */
    public function value($p, Frequency $pf)
    {
        $noy = 1.0 / $pf->termsInAYear();

        return $this->interest($p, $noy);
    }

    /**
     * @param float    $p Principal
     * @param DateTime $sd
     * @param DateTime $fd
     * @return float|null
     */
    public function over($p, DateTime $sd, DateTime $fd)
    {
        $noy = Frequency::dayDiff($sd, $fd) / Frequency::daysInYear();

        return $this->interest($p, $noy);
    }

    /**
     * @param float $p Principal
     * @param float $noy Number of years
     * @return float|null
     */
    private function interest($p, $noy)
    {
        $rt = null;

        if($this->compoundingFrequency != null)
        {
            $ip = null;

            if($this->compoundingFrequency instanceof Continuous)
            {
                $ip = $p * pow(M_E, ($this->annualInterestRate * $noy));
            }
            else
            {
                if($noy > 0)
                {
                    $ncy = $this->compoundingFrequency->termsInAYear();
                    $ip = $p * pow((1 + ($this->annualInterestRate / $ncy)), ($ncy * $noy));
                }
                else
                {
                    $ip = $p;
                }
            }

            $rt = round($ip - $p, self::ROUNDING_PRECISION);
        }
        else
        {
            $rt = $this->simple($p, $noy);
        }

        return $rt;
    }
}