<?php

namespace DPP\Financial\Time;

use \DateTime as DateTime;
use \DateInterval as DateInterval;

use DPP\Financial\FinancialException as FinancialException;

/**
 * Class Frequency
 *
 * @package DPP\Financial\Time
 */
abstract class Frequency
{
    private $id;

    private $name;
    private $shortName;

    /**
     * @var DateInterval
     */
    protected $lastDateInterval;

    /**
     * Frequency constructor.
     *
     * @param int $i ID
     * @param string $n Name
     * @param string $sn Short name
     */
    public function __construct($i, $n, $sn)
    {
        $this->id = $i;

        $this->name = $n;
        $this->shortName = $sn;

        $this->lastDateInterval;
    }

    /**
     * @return self[]
     */
    public static final function GetList()
    {
        $rt = array();

        $rt[] = new Continuous();
        $rt[] = new Daily();
        $rt[] = new Business();
        $rt[] = new Weekly();
        $rt[] = new BiWeekly();
        $rt[] = new SemiMonthly();
        $rt[] = new Monthly();
        $rt[] = new Quarterly();
        $rt[] = new SemiAnnually();
        $rt[] = new Annually();

        return $rt;
    }

    /**
     * @param string $n The name of the freqency to create
     * @return Frequency
     * @throws FinancialException
     */
    public static final function Create($n)
    {
        $rt = null;
        $rst = self::GetList();

        foreach($rst as $r)
        {
            if($r->shortName() == $n)
            {
                $rt = $r;
                break;
            }
        }

        if($rt == null)
        {
            throw new FinancialException(sprintf("Unable to create type '%s'", $n));
        }

        return $rt;
    }

    /**
     * @param DateTime $dt
     * @return DateTime
     * @throws FinancialException
     */
    public static final function CopyDate(DateTime $dt)
    {
        $y = $dt->format("Y");
        $m = $dt->format("m");
        $d = $dt->format("d");

        if($y >= 10000)
        {
            throw new FinancialException("Unable to use PHP to do this");
        }

        $rt = DateTime::createFromFormat("Y-m-d", sprintf("%s-%s-%s", $y, $m, $d), $dt->getTimezone());

        if($rt === false)
        {
            throw new FinancialException("Invalid date format");
        }

        $rt->setTime(0, 0, 0);

        return $rt;
    }

    /**
     * @param DateTime $fdt
     * @param DateTime $sdt
     * @return int
     */
    public static final function DayDiff(DateTime $fdt, DateTime $sdt)
    {
        return intval($fdt->diff($sdt)->format("%a"));
    }

    /**
     * @param DateTime $fdt
     * @param DateTime $sdt
     * @return int
     */
    public static final function YearDiff(DateTime $fdt, DateTime $sdt)
    {
        return intval($fdt->diff($sdt)->format("%y"));
    }

    /**
     * @return int
     */
    public static final function DaysInYear()
    {
        return 365;
    }

    public static final function GenerateSchedule(DateTime $sd, Frequency $f, $nt)
    {
        $rt = array();
        $nd = Frequency::CopyDate($sd);
        $nt = intval($nt);

        while($nt > 0)
        {
            $rt[] = $nd;
            $nd = $f->getNextDate($nd);

            $nt--;
        }

        return $rt;
    }

    /**
     * @return int
     */
    public function id()
    {
        return $this->id;
    }

    /**
     * @return string
     */
    public function name()
    {
        return $this->name;
    }

    /**
     * @return string
     */
    public function shortName()
    {
        return $this->shortName;
    }

    /**
     * @param Frequency $f
     * @return float|int|null
     */
    public function diffWith(Frequency $f)
    {
        $rt = null;

        $tpo = $this->partOfYear();
        $fpo = $f->partOfYear();

        if($fpo > 0.0)
        {
            $rt = $tpo / $fpo;
        }

        return $rt;
    }

    /**
     * @param DateTime $dt
     * @return DateTime
     * @throws FinancialException
     */
    protected function copy(DateTime $dt)
    {
        return self::CopyDate($dt);
    }

    /**
     * @return float
     */
    public abstract function partOfYear();

    /**
     * @param DateTime $dt
     * @return DateTime
     */
    public abstract function getNextDate(DateTime $dt);

    /**
     * @return int
     */
    public abstract function termsInAYear();

    /**
     * @return int
     */
    public abstract function daysInTerm();
}