<?php

namespace DPP\Financial\Time;

use \DateTime as DateTime;
use \DateInterval as DateInterval;

use DPP\Financial\FinancialException as FinancialException;

use DPP\Financial\Time\Holiday\Holiday as Holiday;

/**
 * Class Business
 *
 * @package DPP\Amortization\Time
 */
class Business extends Frequency
{
    private $holidays;
    private $weekends;

    /**
     * Business constructor.
     */
    public function __construct()
    {
        parent::__construct(10, "Business", "business");

        $this->holidays = Holiday::GetList();
        $this->weekends = array(0, 6);
    }

    /**
     * @return float|int
     */
    public function partOfYear()
    {
        return (1 / 365);
    }

    /**
     * @param DateTime $dt
     * @return DateTime
     * @throws FinancialException
     */
    public function getNextDate(DateTime $dt)
    {
        $rt = $this->copy($dt);
        $rt->add(DateInterval::createFromDateString("1 day"));

        while($this->isWeekend($rt) || $this->isHoliday($rt))
        {
            $rt->add(DateInterval::createFromDateString("1 day"));
        }

        return $rt;
    }

    /**
     * @return float|int
     */
    public function termsInAYear()
    {
        return (365 - (52 * 2) - count($this->holidays));
    }

    /**
     * @return int
     */
    public function daysInTerm()
    {
        return 5;
    }

    /**
     * @param DateTime $dt
     * @return bool
     */
    private function isWeekend(DateTime $dt)
    {
        $dow = $dt->format("w");
        return (in_array($dow, $this->weekends));
    }

    /**
     * @param DateTime $dt
     * @return bool
     */
    private function isHoliday(DateTime $dt)
    {
        $rt = false;

        foreach($this->holidays as $h)
        {
            if($h->isHoliday($dt))
            {
                $rt = true;
                break;
            }
        }

        return $rt;
    }
}