<?php

namespace DPP\Financial;

use \DateTime as DateTime;

use DPP\Financial\Time\Frequency as Frequency;

/**
 * Class ScheduledPayment
 *
 * @package DPP\Financial
 */
class ScheduledPayment
{
    private $paymentDate;
    private $periodStartDate;
    private $annualInterestRate;
    private $paymentAmount;

    private $totalAmountAfterPayment;
    private $interestAmountPaid;
    private $principalAmountPaid;

    private $locked;

    /**
     * ScheduledPayment constructor.
     *
     * @param DateTime $pd
     * @param DateTime $psd
     * @param float    $air
     * @param null     $pa
     */
    public function __construct(DateTime $pd, DateTime $psd, $air, $pa = null)
    {
        $this->paymentDate = $pd;
        $this->periodStartDate = $psd;
        $this->annualInterestRate = $air;
        $this->paymentAmount = $pa;

        $this->totalAmountAfterPayment = null;
        $this->interestAmountPaid = null;
        $this->principalAmountPaid = null;

        $this->locked = false;
    }

    /**
     * @return DateTime
     */
    public function date()
    {
        return $this->paymentDate;
    }

    /**
     * @return float
     */
    public function part()
    {
        return $this->length() / 365;
    }

    /**
     * @return float
     */
    public function principal()
    {
        return $this->principalAmountPaid;
    }

    /**
     * @return float
     */
    public function interest()
    {
        return $this->interestAmountPaid;
    }

    /**
     * @return float
     */
    public function balance()
    {
        return $this->totalAmountAfterPayment;
    }

    /**
     * @return float
     */
    public function amount()
    {
        return $this->principalAmountPaid + $this->interestAmountPaid;
    }

    /**
     * @return int
     */
    public function length()
    {
        return Frequency::DayDiff($this->paymentDate, $this->periodStartDate);
    }

    /**
     * @return DateTime
     */
    public function start()
    {
        return $this->periodStartDate;
    }

    /**
     * @return DateTime
     */
    public function end()
    {
        return $this->paymentDate;
    }

    /**
     * @return float
     */
    public function annual()
    {
        return $this->annualInterestRate;
    }

    /**
     * @return float
     */
    public function payment()
    {
        return $this->paymentAmount;
    }

    /**
     * @param float $pp
     * @param float $ip
     * @param float $tap
     */
    public function lock($pp, $ip, $tap)
    {
        if(!$this->locked)
        {
            $this->principalAmountPaid = $pp;
            $this->interestAmountPaid = $ip;
            $this->totalAmountAfterPayment = $tap;

            $this->locked = true;
        }
    }

    public function isOnTime(DateTime $dt)
    {
        return ($dt <= $this->paymentDate);
    }

    /**
     * @param ScheduledPayment $sp
     * @return int
     */
    public function dayDiff(ScheduledPayment $sp)
    {
        return Frequency::DayDiff($this->paymentDate, $sp->paymentDate);
    }

    /**
     * @return array
     */
    public function toArray()
    {
        $rt = array(
            'payment_date'              => $this->paymentDate->format("Y-m-d"),
            'principal_amount'          => $this->principalAmountPaid,
            'interest_amount'           => $this->interestAmountPaid,
            'total_amount_remaining'    => $this->totalAmountAfterPayment,
            'amount_this_payment'       => $this->principalAmountPaid + $this->interestAmountPaid
        );

        return $rt;
    }
}