<?php

namespace DPP\Financial;

use DPP\Financial\Time\Frequency as Frequency;

/**
 * Class PresentValueOfAnnuity
 *
 * @package DPP\Financial
 */
class PresentValueOfAnnuity
{
    private $frequency;
    private $annualInterestRate;
    private $numberOfPeriods;
    private $compoundFrequency;

    /**
     * PresentValueOfAnnuity constructor.
     *
     * @param Frequency      $f
     * @param float          $air
     * @param int            $nop
     * @param Frequency|null $cpf
     */
    public function __construct(Frequency $f, $air, $nop, Frequency $cpf = null)
    {
        $this->frequency = $f;
        $this->annualInterestRate = $air;
        $this->numberOfPeriods = $nop;
        $this->compoundFrequency = $cpf;
    }

    /**
     * @param float $pp
     * @return float
     */
    public function getTotalPayment($pp)
    {
        $rpp = $this->getPeriodicRate();
        $rt = $pp * ((1 - pow((1 + $rpp), (-1 * $this->numberOfPeriods))) / ($rpp));

        return round($rt, Interest::ROUNDING_PRECISION);
    }

    /**
     * @param float $toa
     * @return float
     */
    public function getPeriodicPayment($toa)
    {
        $rpp = $this->getPeriodicRate();
        $npp = pow((1 + $rpp), (-1 * $this->numberOfPeriods));
        $rt = ($toa * $rpp) / (1 - $npp);

        return round($rt, Interest::ROUNDING_PRECISION);
    }

    /**
     * @return float
     */
    private function getPeriodicRate()
    {
        return $this->getEAR() * (1 / $this->frequency->termsInAYear());
    }

    /**
     * @return float
     */
    private function getEAR()
    {
        $rt = $this->annualInterestRate;

        if($this->compoundFrequency != null)
        {
            if($this->compoundFrequency->shortName() == "continuous")
            {
                $rt = pow(M_E, $this->annualInterestRate) - 1;
            }
            else
            {
                $ty = $this->compoundFrequency->termsInAYear();
                $rt = pow((1 + ($this->annualInterestRate / $ty)), $ty) - 1;
            }
        }

        return $rt;
    }
}