<?php

namespace DPP\Financial\Amortization;

use \DateTime as DateTime;

use DPP\Financial\FinancialException as FinancialException;
use DPP\Financial\PresentValueOfAnnuity as PresentValueOfAnnuity;
use DPP\Financial\Interest as Interest;
use DPP\Financial\ScheduledPayment as ScheduledPayment;

use DPP\Financial\Time\Frequency as Frequency;

/**
 * Class NormalAmortizationCalculator
 *
 * @package DPP\Financial\Amortization
 */
class NormalAmortizationCalculator extends AmortizationCalculator
{
    private $lastTotal;
    private $increasingCount;

    /**
     * NormalAmortizationCalculator constructor.
     */
    public function __construct()
    {
        parent::__construct();

        $this->lastTotal = null;
        $this->increasingCount = 0;
    }

    /**
     * @param DateTime[] $ds
     * @return ScheduledPayment[]
     * @throws FinancialException
     */
    protected function fixedSchedule(array $ds)
    {
        $rt = array();
        $ft = $this->totalAmount;

        $np = count($ds);
        $oa = $ds[0]->annual();
        $pa = $this->paymentAmount;

        for($i=0; $i<$np; $i++)
        {
            $d = $ds[$i];
            $da = $d->annual();
            $ip = null;
            $dt = $d->date();
            $tpa = null;

            if(($this->incrementType != null) && ($oa != $da))
            {
                $nop = $this->amortizationTermCount ?: ($np - $i) ?: null;

                $pva = new PresentValueOfAnnuity($this->paymentFrequency, $da, $nop, $this->compoundFrequency);
                $pa = $pva->getPeriodicPayment($ft);

                $oa = $da;
            }

            $pp = null;

            if(($this->initialDate < $dt) && ($i == 0))
            {
                $io = new Interest($d->annual(), $this->compoundFrequency);
                $ip = $io->over($ft, $this->initialDate, $dt);

                if($ip > $pa)
                {
                    $tpa = $ip;
                }
            }
            else
            {
                $io = new Interest($d->annual(), $this->compoundFrequency);
                $ip = $io->value($ft, $this->paymentFrequency);
            }

            if($ft >= $pa)
            {
                $pp = round(($tpa ?: $pa) - $ip, Interest::ROUNDING_PRECISION);
            }
            else
            {
                $pp = round($ft, Interest::ROUNDING_PRECISION);
            }

            $ft = round(($ft - $pp), Interest::ROUNDING_PRECISION);

            // On last payment, add to principal and zero out the remaining balance
            if(count($rt) == (count($ds) - 1))
            {
                $pp += $ft;
                $ft = 0.0;
            }

            $d->lock($pp, $ip, $ft);
            $rt[] = $d;

            if($ft <= 0.0)
            {
                break;
            }

            $this->determineRunaway($ft);
        }

        return $rt;
    }

    /**
     * @return ScheduledPayment[]
     * @throws FinancialException
     */
    protected function adHocSchedule()
    {
        $rt = array();

        if($this->incrementType != null)
        {
            throw new FinancialException("Unable to calculate ad-hoc ARM schedule");
        }

        $od = Frequency::CopyDate($this->initialDate);
        $nd = Frequency::CopyDate($this->firstPaymentDate);
        $ft = $this->totalAmount;

        while($ft > 0.0)
        {
            $io = new Interest($this->annualInterestRate, $this->compoundFrequency);
            $ip = $io->value($ft, $this->paymentFrequency);

            $pp = round($this->paymentAmount - $ip, Interest::ROUNDING_PRECISION);

            if($ft < $this->paymentAmount)
            {
                $pp = round($ft, Interest::ROUNDING_PRECISION);
            }

            $ft = round(($ft - $pp), Interest::ROUNDING_PRECISION);
            $ns = new ScheduledPayment($nd, $od, $this->annualInterestRate);

            $ns->lock($pp, $ip, $ft);
            $rt[] = $ns;

            $od = $nd;
            $nd = $this->paymentFrequency->getNextDate($nd);

            $this->determineRunaway($ft);
        }

        $this->termCount = count($rt);

        return $rt;
    }

    /**
     * @param float $ft
     * @throws FinancialException
     */
    private function determineRunaway($ft)
    {
        if($this->lastTotal != null)
        {
            if($ft >= $this->lastTotal)
            {
                $this->increasingCount++;
            }

            if($this->increasingCount > 10)
            {
                throw new FinancialException("Runaway Interest, payment too small");
            }
        }

        $this->lastTotal = $ft;
    }
}