<?php

namespace DPP\Financial\Amortization;

use \DateTime as DateTime;

use DPP\Financial\FinancialException as FinancialException;
use DPP\Financial\Interest as Interest;
use DPP\Financial\ScheduledPayment as ScheduledPayment;

use DPP\Financial\Time\Frequency as Frequency;

/**
 * Class FixedPrincipalAmortizationCalculator
 *
 * @package DPP\Financial\Amortization
 */
class FixedPrincipalAmortizationCalculator extends AmortizationCalculator
{
    /**
     * FixedPrincipalAmortizationCalculator constructor.s
     */
    public function __construct()
    {
        parent::__construct();
    }

    /**
     * @param DateTime[] $ds
     * @return ScheduledPayment[]
     */
    protected function fixedSchedule(array $ds)
    {
        $rt = array();
        $ft = $this->totalAmount;
        $pa = $this->paymentAmount;

        for($i=0; $i<count($ds); $i++)
        {
            $d = $ds[$i];
            $ip = null;

            $io = new Interest($d->annual(), $this->compoundFrequency);
            $ip = $io->value($ft, $this->paymentFrequency);

            $pm = round($pa + $ip, Interest::ROUNDING_PRECISION);

            if($ft < $pm)
            {
                $ip = 0;
                $pa = $ft;
            }

            $ft = $ft - $pa;

            // On last payment, add to principal and zero out the remaining balance
            if(count($rt) == (count($ds) - 1))
            {
                $pa += $ft;
                $ft = 0.0;
            }

            $d->lock($pa, $ip, $ft);
            $rt[] = $d;

            if($ft <= 0.0)
            {
                break;
            }
        }

        return $rt;
    }

    /**
     * @return ScheduledPayment[]
     * @throws FinancialException
     */
    protected function adHocSchedule()
    {
        $rt = array();

        $nd = Frequency::CopyDate($this->firstPaymentDate);
        $ft = $this->totalAmount;
        $pa = $this->paymentAmount;
        $od = null;

        while($ft > 0.0)
        {
            $ip = null;

            $io = new Interest($this->annualInterestRate, $this->compoundFrequency);
            $ip = $io->value($ft, $this->paymentFrequency);

            if($ft < $this->paymentAmount)
            {
                $ip = 0;
                $pa = $ft;
            }

            $ft = $ft - $pa;
            $ns = new ScheduledPayment($nd, $od, $this->annualInterestRate);

            $ns->lock($pa, $ip, $ft);
            $rt[] = $ns;

            $od = $nd;
            $nd = $this->paymentFrequency->getNextDate($nd);
        }

        return $rt;
    }
}