<?php

namespace DPP\Financial\Amortization;

use \DateTime as DateTime;

use DPP\Financial\FinancialException as FinancialException;
use DPP\Financial\IncrementType as IncrementType;
use DPP\Financial\PresentValueOfAnnuity as PresentValueOfAnnuity;
use DPP\Financial\ScheduledPayment as ScheduledPayment;
use DPP\Financial\Schedule as Schedule;

use DPP\Financial\Time\Frequency as Frequency;

/**
 * Class AmortizationCalculator
 *
 * @package DPP\Financial\Amortization
 */
abstract class AmortizationCalculator
{
    const CURRENCY = "$";
    const AD_HOC = "ad-hoc";
    const STANDARD = "standard";

    /**
     * @var Frequency
     */
    protected $paymentFrequency;
    protected $totalAmount;
    protected $annualInterestRate;
    protected $termCount;
    protected $amortizationTermCount;
    protected $compoundFrequency;
    protected $paymentAmount;

    /**
     * @var IncrementType
     */
    protected $incrementType;
    protected $initialDate;
    protected $firstPaymentDate;
    protected $lastPaymentDate;
    protected $incrementValue;

    private $mode;

    /**
     * AmortizationCalculator constructor.
     */
    public function __construct()
    {
        $this->paymentFrequency;
        $this->totalAmount = null;
        $this->annualInterestRate = null;
        $this->termCount = null;
        $this->amortizationTermCount = null;
        $this->compoundFrequency = null;
        $this->paymentAmount = null;

        $this->incrementType;
        $this->initialDate = null;
        $this->firstPaymentDate = null;
        $this->lastPaymentDate = null;
        $this->incrementValue = null;

        $this->mode = null;
    }

    /**
     * @param string $ms The string description of the method desired
     * @return Method The final method object
     * @throws FinancialException
     */
    public static final function GetMethod($ms)
    {
        $rt = null;
        $mtd = Method::GetList();

        foreach($mtd as $m)
        {
            if($m->shortName() == $ms)
            {
                $rt = $m;
                break;
            }
        }

        if($rt == null)
        {
            throw new FinancialException(sprintf("Method string '%s' is invalid", $ms));
        }

        return $rt;
    }

    /**
     * @param string $fs
     * @return Frequency|null
     * @throws FinancialException
     */
    public static final function GetFrequency($fs)
    {
        $rt = null;
        $frq = Frequency::GetList();

        foreach($frq as $f)
        {
            if($f->shortName() == $fs)
            {
                $rt = $f;
                break;
            }
        }

        if($rt == null)
        {
            throw new FinancialException(sprintf("Frequency string '%s' is invalid", $fs));
        }

        return $rt;
    }

    /**
     * @param string $tp
     * @return IncrementType|null
     * @throws FinancialException
     */
    public static final function GetIncrementType($tp)
    {
        $rt = null;
        $its = IncrementType::GetList();

        foreach($its as $i)
        {
            if($i->shortName() == $tp)
            {
                $rt = $i;
                break;
            }
        }

        if($rt == null)
        {
            throw new FinancialException(sprintf("Increment type string '%s' is invalid", $tp));
        }

        return $rt;
    }

    /**
     * @param Method             $mtd Amortization method
     * @param float              $amt Initial total amount
     * @param float              $air Annual interest rate
     * @param Frequency          $pf Payment frequency
     * @param DateTime|null      $id Initial date
     * @param DateTime|null      $fpd First payment date
     * @param DateTime|null      $lpd Last payment date
     * @param int|null           $tc Term count
     * @param float|null         $pa Payment amount
     * @param Frequency|null     $cp Compounding frequency
     * @param IncrementType|null $it Increment type
     * @param float|null         $iv Increment value
     * @param int|null           $atc Amortization term count
     * @return AmortizationCalculator|null
     * @throws FinancialException
     */
    public static final function Create(Method $mtd, $amt, $air, Frequency $pf, DateTime $id = null, DateTime $fpd = null, DateTime $lpd = null, $tc = null, $pa = null, Frequency $cp = null, IncrementType $it = null, $iv = null, $atc = null)
    {
        $nm = $mtd->className();
        $rt = null;

        switch($nm)
        {
            case "NormalAmortizationCalculator": { $rt = new NormalAmortizationCalculator(); break; }
            case "FixedPrincipalAmortizationCalculator": { $rt = new FixedPrincipalAmortizationCalculator(); break; }
            case "InterestOnlyAmortizationCalculator": { $rt = new InterestOnlyAmortizationCalculator(); break; }
            default: { throw new FinancialException("Invalid method"); break; }
        }

        $rt->totalAmount = $amt;
        $rt->annualInterestRate = $air;
        $rt->paymentFrequency = $pf;
        $rt->termCount = $tc;
        $rt->amortizationTermCount = $atc;
        $rt->compoundFrequency = $cp;
        $rt->paymentAmount = $pa;

        if($id != null)
        {
            $id->setTime(0, 0, 0);
        }

        if($fpd != null)
        {
            $fpd->setTime(0, 0, 0);
        }

        if($lpd != null)
        {
            $lpd->setTime(0, 0, 0);
        }

        if(($id != null) && ($fpd == null))
        {
            $rt->firstPaymentDate = $rt->initialDate;
        }

        $rt->initialDate = $id;
        $rt->firstPaymentDate = $fpd;
        $rt->lastPaymentDate = $lpd;

        $rt->incrementType = $it;
        $rt->incrementValue = $iv;

        $nd = ($pa != null) && ($fpd != null);
        $fd = ($tc == null) && (($fpd != null) && ($lpd != null));
        $ft = ($tc != null);

        if((!$fd) && (!$ft) && (!$nd))
        {
            throw new FinancialException("Insufficient information to calculate number of terms");
        }

        if($fd || $ft)
        {
            $rt->mode = self::STANDARD;
        }
        else
        {
            $rt->mode = self::AD_HOC;
        }

        if(($id != null) && ($fpd != null))
        {
            if($id > $fpd)
            {
                throw new FinancialException("Initial date cannot come after first payment date");
            }
        }

        return $rt;
    }

    public static function CreateFromSchedule(Schedule $s)
    {
        
    }

    /**
     * @return int|null
     */
    public function getTermCount()
    {
        return $this->termCount;
    }

    /**
     * @return DateTime[]
     * @throws FinancialException
     */
    public function getSchedule()
    {
        $dts = array();

        if($this->initialDate != $this->firstPaymentDate)
        {
            $dts[] = new ScheduledPayment(Frequency::CopyDate($this->firstPaymentDate), Frequency::CopyDate($this->initialDate), $this->annualInterestRate);
        }
        else
        {
            $dts[] = new ScheduledPayment(Frequency::CopyDate($this->firstPaymentDate), Frequency::CopyDate($this->firstPaymentDate), $this->annualInterestRate);
        }

        $fd = Frequency::CopyDate($this->firstPaymentDate);
        $pfd = Frequency::CopyDate($this->firstPaymentDate);

        if($this->paymentAmount == null)
        {
            $tat = $this->totalAmount;
            $nop = $this->amortizationTermCount ?: $this->termCount ?: null;

            $pva = new PresentValueOfAnnuity($this->paymentFrequency, $this->annualInterestRate, $nop, $this->compoundFrequency);
            $this->paymentAmount = $pva->getPeriodicPayment($tat);
        }

        $tc = $this->termCount ?: null;

        if($tc != null)
        {
            $ntc = $tc - count($dts);

            while($ntc > 0)
            {
                $ar = $this->annualInterestRate;
                $pd = $this->paymentFrequency->getNextDate($fd);

                if($this->incrementType != null)
                {
                    $ydf = Frequency::YearDiff($pfd, $pd);
                    $ar += floor($ydf / $this->incrementType->years()) * $this->incrementValue;
                }

                $dts[] = new ScheduledPayment($pd, $fd, $ar);

                $fd = $pd;
                $ntc--;
            }
        }
        else
        {
            while($fd < $this->lastPaymentDate)
            {
                $ar = $this->annualInterestRate;
                $pd = $this->paymentFrequency->getNextDate($fd);

                if($this->incrementType != null)
                {
                    $ydf = Frequency::YearDiff($pfd, $pd);
                    $ar += floor($ydf / $this->incrementType->years()) * $this->incrementValue;
                }

                $rp = $this->paymentFrequency->partOfYear() * $ar;

                $dts[] = new ScheduledPayment($pd, $fd, $rp);
                $fd = $pd;
            }
        }

        if($this->termCount == null)
        {
            $this->termCount = count($dts);
        }

        return $dts;
    }

    /**
     * @return Schedule
     * @throws FinancialException
     */
    public function run()
    {
        $rt = new Schedule(self::CURRENCY);
        $ncs = null;

        if($this->mode == self::AD_HOC)
        {
            $ncs = $this->adHocSchedule();
        }
        else
        {
            $sch = $this->getSchedule();
            $ncs = $this->fixedSchedule($sch);
        }

        $rt->addAll($ncs);
        $rt->lock();

        return $rt;
    }

    /**
     * @param DateTime[] $ds
     * @return ScheduledPayment[]
     */
    protected abstract function fixedSchedule(array $ds);

    /**
     * @return ScheduledPayment[]
     */
    protected abstract function adHocSchedule();
}